using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

/** \file MCExports.cs
 * Imports functions, structs, and enums from the Mobile-C library.
 */

namespace LibMC
{
    /**
     * \brief   Enum for describing the type of an agent.
     * 
     * \note    This enum is pulled directly from the Mobile-C library.
     */
    public enum MC_AgentType_e
    {
        MC_NONE = -1,           /**< Default value to describe unininitialized agent. */
        MC_REMOTE_AGENT = 0,    /**< A remote agent. */
        MC_LOCAL_AGENT,         /**< A local agent. */
        MC_RETURN_AGENT         /**< A returning agent. */
    };

    /**
     * \brief   Enum for describing the status of an agent.
     * 
     * \note    This enum is pulled directly from the Mobile-C library.
     */
    public enum MC_AgentStatus_e
    {
        MC_NO_STATUS = -1, /*!< Default value for uninitialized agent */
        MC_WAIT_CH = 0,    /*!< Waiting to be started */
        MC_WAIT_MESSGSEND, /*!< Finished, waiting to migrate */
        MC_AGENT_ACTIVE,   /*!< Running */
        MC_AGENT_NEUTRAL,  /*!< Not running, but do not flush */
        MC_AGENT_SUSPENDED,/*!< Unused */
        MC_WAIT_FINISHED   /*!< Finished, waiting to be flushed */
    };

    /**
     * \brief   Enum for describing the type of an ACL message.
     * 
     * \note    This enum is pulled directly from the Mobile-C library.
     */
    public enum MC_FipaPerformative_e
    {
        FIPA_ERROR = -1,            /*!< Fipa performative enum value */
        FIPA_ZERO,                  /*!< Fipa performative enum value */
        FIPA_ACCEPT_PROPOSAL,       /*!< Fipa performative enum value */
        FIPA_AGREE,                 /*!< Fipa performative enum value */
        FIPA_CANCEL,                /*!< Fipa performative enum value */
        FIPA_CALL_FOR_PROPOSAL,     /*!< Fipa performative enum value */
        FIPA_CONFIRM,               /*!< Fipa performative enum value */
        FIPA_DISCONFIRM,            /*!< Fipa performative enum value */
        FIPA_FAILURE,               /*!< Fipa performative enum value */
        FIPA_INFORM,                /*!< Fipa performative enum value */
        FIPA_INFORM_IF,             /*!< Fipa performative enum value */
        FIPA_INFORM_REF,            /*!< Fipa performative enum value */
        FIPA_NOT_UNDERSTOOD,        /*!< Fipa performative enum value */
        FIPA_PROPOGATE,             /*!< Fipa performative enum value */
        FIPA_PROPOSE,               /*!< Fipa performative enum value */
        FIPA_PROXY,                 /*!< Fipa performative enum value */
        FIPA_QUERY_IF,              /*!< Fipa performative enum value */
        FIPA_QUERY_REF,             /*!< Fipa performative enum value */
        FIPA_REFUSE,                /*!< Fipa performative enum value */
        FIPA_REJECT_PROPOSAL,       /*!< Fipa performative enum value */
        FIPA_REQUEST,               /*!< Fipa performative enum value */
        FIPA_REQUEST_WHEN,          /*!< Fipa performative enum value */
        FIPA_REQUEST_WHENEVER,      /*!< Fipa performative enum value */
        FIPA_SUBSCRIBE              /*!< Fipa performative enum value */
    };

    public partial class MCAgency
    {
        //private const String mcdll = "C:\\mobilec\\src\\win32\\vcnet2005\\dll debug\\libmcd.dll";
        private const String mcdll = 
#if DEBUG
            "libmcd.dll";
#else
            "libmc.dll";
#endif

        /**
         * \brief   TACOS!
         */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        internal struct MCAgency_t
        {
            int client;
            int server;
            String hostName;                     /*!< Local Hostname */
            String filename;
            int portno;                         /*!< Local port number */
            int portnoc;
            IntPtr mc_platform;  /*!< Local MobileC Platform */
            int default_agentstatus;            /*!< Agency default agent status */
            int threads;                        /*!< flag which determines which threads to start */
            int enable_security;                /*!< Security flag */
            //int stack_size[MC_THREAD_ALL];
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 5)]
            int[] stack_size;
            int last_error;
        };

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        internal struct MCAgencyOptions_t
        {
            int threads;              /*!< Threads to start */
            int default_agent_status; /*!< Default agent status */
            int modified;             /*!< unused */
            int enable_security;      /*!< security enable flag */
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 5)]
            /* Following are some thread stack size options: unix/pthreads only! */
            int[] stack_size; /*!< If the stack size is -1, use system def.*/
        };

        /**
         * \brief     ChOptions structures.
         * 
         * Allows the user to set the shell type and the home directory.
         * 
         * \note    This struct is pulled directly from the Mobile-C library.
         */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        internal struct ChOptions_t
        {
            public int shelltype;        /*!< shell type: CH_REGULARCH or CH_SAFECH */
            /* int chrc; */       /* deprecated, use default value */
            /* char *chrcname; */ /* deprecated, use default value  */
            public String chhome;         /*!< Embedded Ch home directory. if NULL, use Ch home directory for 
                                    standard/professional edition as Embedded Ch home directory */
            /* char *chmtdir; */  /* directory for dynamically loaded libs chmt1.dl, chmt2.dl etc. 
                                    deprecated, use default value*/
        };

        /**
         * \brief     Enum for describing the different threads that Mobile-C uses.
         * 
         * These enums can be used to turn threads on and off before an agency is initialized.
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         */
        public enum MC_ThreadIndex_e
        {
            MC_THREAD_DF = 0,   /*!< Directory Facilitator */
            MC_THREAD_AMS,    /*!< Agent Managment system */
            MC_THREAD_ACC,    /*!< Agency communications */
            MC_THREAD_CP,     /*!< Command Prompt */
            MC_THREAD_AGENT,  /*!< Agent threads */
            MC_THREAD_ALL
        };

        /**
         * \brief     Available commands for MC_Steer
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         */
        public enum MC_SteerCommand_e {
          MC_RUN = 0,     /*!< Continue the algorithm */
          MC_SUSPEND,     /*!< Suspend/pause the algorithm */
          MC_RESTART,     /*!< Restart the algorithm from the beginning */
          MC_STOP         /*!< Stop the algorithm */
        };

        /**
         * \brief     MobileC system signals.
         *
         * Each signal is activated after the corresponding action. i.e. The
         * 'MC_RECV_MESSAGE' signal is activated after a message is received.
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         *
         * \sa        MC_WaitSignal(), MC_ResetSignal()
         */
        enum MC_Signal_e{
            MC_NO_SIGNAL       = 0x00,
            MC_RECV_CONNECTION = 0x01,
            MC_RECV_MESSAGE    = 0x02,
            MC_RECV_AGENT      = 0x04,
            MC_RECV_RETURN     = 0x08,
            MC_EXEC_AGENT      = 0x0F,
            MC_ALL_SIGNALS     = 0x10
        };

        /*
         * Functions to import from libmc.dll
         */
        // Basic agency functions
        [DllImport(mcdll, EntryPoint = "MC_Initialize")]
        private static extern IntPtr _MC_Initialize(int port, ref MCAgencyOptions_t options);
        [DllImport(mcdll, EntryPoint = "MC_End")]
        private static extern int _MC_End(IntPtr agency);//ref MCAgency_t agency);
        [DllImport(mcdll, EntryPoint = "MC_ChInitializeOptions")]
        private static extern int _MC_ChInitializeOptions(IntPtr agency, ChOptions_t options);
        [DllImport(mcdll, EntryPoint = "MC_InitializeAgencyOptions")]
        private static extern int _MC_InitializeAgencyOptions(ref MCAgencyOptions_t options);
        [DllImport(mcdll, EntryPoint = "MC_SetThreadsAllOn")]
        private static extern int _MC_SetThreadsAllOn(ref MCAgencyOptions_t options);
        [DllImport(mcdll, EntryPoint = "MC_SetThreadsAllOff")]
        private static extern int _MC_SetThreadsAllOff(ref MCAgencyOptions_t options);
        [DllImport(mcdll, EntryPoint = "MC_SetThreadOn")]
        private static extern int _MC_SetThreadOn(ref MCAgencyOptions_t options, MC_ThreadIndex_e index);
        [DllImport(mcdll, EntryPoint = "MC_SetThreadOff")]
        private static extern int _MC_SetThreadOff(ref MCAgencyOptions_t options, MC_ThreadIndex_e index);
        [DllImport(mcdll, EntryPoint = "MC_HaltAgency")]
        private static extern int _MC_HaltAgency(IntPtr agency);
        [DllImport(mcdll, EntryPoint = "MC_ResumeAgency")]
        private static extern int _MC_ResumeAgency(IntPtr agency);
        [DllImport(mcdll, EntryPoint = "MC_SetDefaultAgentStatus")]
        private static extern int _MC_SetDefaultAgentStatus(IntPtr agency, MC_AgentStatus_e status);
        [DllImport(mcdll, EntryPoint = "MC_WaitRetrieveAgent")]
        private static extern IntPtr _MC_WaitRetrieveAgent(IntPtr agency);
        [DllImport(mcdll, EntryPoint = "MC_WaitAgent")]
        private static extern int _MC_WaitAgent(IntPtr agency);

        // Migration functions
        [DllImport(mcdll, EntryPoint = "MC_SendAgentMigrationMessageFile")]
        private static extern int _MC_SendAgentMigrationMessageFile(IntPtr agency, String filename, String hostname, int port);
        [DllImport(mcdll, EntryPoint = "MC_SendAgentMigrationMessage")]
        private static extern int _MC_SendAgentMigrationMessage(IntPtr agency, String message, String hostname, int port);

        // Signals and threading
        [DllImport(mcdll, EntryPoint = "MC_CondBroadcast")]
        private static extern int _MC_CondBroadcast(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_CondSignal")]
        private static extern int _MC_CondSignal(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_CondReset")]
        private static extern int _MC_CondReset(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_CondWait")]
        private static extern int _MC_CondWait(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_MutexLock")]
        private static extern int _MC_MutexLock(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_MutexUnlock")]
        private static extern int _MC_MutexUnlock(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_SemaphorePost")]
        private static extern int _MC_SemaphorePost(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_SemaphoreWait")]
        private static extern int _MC_SemaphoreWait(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_ResetSignal")]
        private static extern int _MC_ResetSignal(IntPtr agency);
        [DllImport(mcdll, EntryPoint = "MC_SyncDelete")]
        private static extern int _MC_SyncDelete(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_SyncInit")]
        private static extern int _MC_SyncInit(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_WaitSignal")]
        private static extern int _MC_WaitSignal(IntPtr agency, int signals);
        [DllImport(mcdll, EntryPoint = "MC_BarrierDelete")]
        private static extern int _MC_BarrierDelete(IntPtr agency, int id);
        [DllImport(mcdll, EntryPoint = "MC_BarrierInit")]
        private static extern int _MC_BarrierInit(IntPtr agency, int id, int num_procs);

        //Steering functions
        [DllImport(mcdll, EntryPoint = "MC_SteerControl")]
        public static extern MC_SteerCommand_e _MC_SteerControl();
        /*[DllImport(dllfile, EntryPoint = "MC_Steer")]
        private static extern int _MC_Steer(IntPtr agency, int (*funcptr)(void* data), void *arg);*/

        // Services
        [DllImport(mcdll, EntryPoint = "MC_RegisterService")]
        private static extern int _MC_RegisterService(IntPtr agency, IntPtr agent, int agentID, String agentName, String[] serviceNames, int numServices);
        /*[DllImport(dllfile, EntryPoint = "MC_SearchForService")]
        private static extern int _MC_SearchForService(IntPtr agency, String searchString, char*** agentNames, char*** serviceNames, int** agentIDs, int* numResults);*/

        // Agent-related agency functions
        [DllImport(mcdll, EntryPoint = "MC_AddAgent")]
        private static extern int _MC_AddAgent(IntPtr agency, IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_DeleteAgent")]
        internal static extern int _MC_DeleteAgent(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_FindAgentByName")]
        private static extern IntPtr _MC_FindAgentByName(IntPtr agency, String name);
        [DllImport(mcdll, EntryPoint = "MC_FindAgentByID")]
        private static extern IntPtr _MC_FindAgentByID(IntPtr agency, int ID);
        [DllImport(mcdll, EntryPoint = "MC_RetrieveAgent")]
        private static extern IntPtr _MC_RetrieveAgent(IntPtr agency);

        // Agent member functions
        [DllImport(mcdll, EntryPoint = "MC_GetAgentID")]
        internal static extern int _MC_GetAgentID(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_GetAgentName")]
        internal static extern String _MC_GetAgentName(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_GetAgentNumTasks")]
        internal static extern int _MC_GetAgentNumTasks(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_GetAgentStatus")]
        internal static extern MC_AgentStatus_e _MC_GetAgentStatus(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_GetAgentType")]
        internal static extern MC_AgentType_e _MC_GetAgentType(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_GetAgentXMLString")]
        internal static extern String _MC_GetAgentXMLString(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_PrintAgentCode")]
        internal static extern int _MC_PrintAgentCode(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_RetrieveAgentCode")]
        internal static extern String _MC_RetrieveAgentCode(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_SetAgentStatus")]
        internal static extern int _MC_SetAgentStatus(IntPtr agent, MC_AgentStatus_e status);
        [DllImport(mcdll, EntryPoint = "MC_TerminateAgent")]
        internal static extern int _MC_TerminateAgent(IntPtr agent);
        /*[DllImport(dllfile, EntryPoint = "MC_CallAgentFunc")]
        internal static extern int _MC_CallAgentFunc(IntPtr agent, String funcName, void* returnVal, void* varg);*/
        /*[DllImport(dllfile, EntryPoint = "MC_GetAgentExecEngine")]
        internal static extern void* _MC_GetAgentExecEngine(IntPtr agent);*/
        /*[DllImport(dllfile, EntryPoint = "MC_GetAgentReturnData")]
        internal static extern int _MC_GetAgentReturnData(IntPtr agent, int task_num, void **data, int *dim, int **extent);*/

        // If the return is an Intptr, it's really a fipa_acl_message_s pointer.
        // If an argument is IntPtr, it's a pointer to a fipa_acl_message_s
        // ACL functions
        [DllImport(mcdll, EntryPoint = "MC_AclNew")]
        internal static extern IntPtr _MC_AclNew();
        [DllImport(mcdll, EntryPoint = "MC_AclPost")]
        internal static extern int _MC_AclPost(IntPtr agent, IntPtr message);
        [DllImport(mcdll, EntryPoint = "MC_AclReply")]
        internal static extern IntPtr _MC_AclReply(IntPtr acl_message);
        [DllImport(mcdll, EntryPoint = "MC_AclRetrieve")]
        internal static extern IntPtr _MC_AclRetrieve(IntPtr agent);
        [DllImport(mcdll, EntryPoint = "MC_AclSend")]
        internal static extern int _MC_AclSend(IntPtr agency, IntPtr acl_message);
        [DllImport(mcdll, EntryPoint = "MC_AclWaitRetrieve")]
        internal static extern IntPtr _MC_AclWaitRetrieve(IntPtr agent);

        /* ACL Helper Functions Here */
        [DllImport(mcdll, EntryPoint = "MC_AclSetPerformative")]
        internal static extern int _MC_AclSetPerformative(IntPtr acl, MC_FipaPerformative_e performative);
        [DllImport(mcdll, EntryPoint = "MC_AclSetSender")]
        internal static extern int _MC_AclSetSender(IntPtr acl, String name, String address);
        [DllImport(mcdll, EntryPoint = "MC_AclAddReceiver")]
        internal static extern int _MC_AclAddReceiver(IntPtr acl, String name, String address);
        [DllImport(mcdll, EntryPoint = "MC_AclAddReplyTo")]
        internal static extern int _MC_AclAddReplyTo(IntPtr acl, String name, String address);
        [DllImport(mcdll, EntryPoint = "MC_AclSetContent")]
        internal static extern int _MC_AclSetContent(IntPtr acl, String content);
        [DllImport(mcdll, EntryPoint = "MC_AclDestroy")]
        internal static extern int _MC_AclDestroy(IntPtr acl);

        /*
         * List of functions to import
        //struct fipa_acl_message_s* MC_AclNew(void);
        //int MC_AclPost(MCAgent_t agent, struct fipa_acl_message_s* message);
        //struct fipa_acl_message_s* MC_AclReply(struct fipa_acl_message_s* acl_message);
        //struct fipa_acl_message_s* MC_AclRetrieve(MCAgent_t agent);
        //int MC_AclSend(MCAgency_t attr, struct fipa_acl_message_s* acl);
        //struct fipa_acl_message_s* MC_AclWaitRetrieve(MCAgent_t agent);
        //int MC_AddAgent(MCAgency_t attr, MCAgent_t agent);
        //int MC_BarrierDelete(MCAgency_t attr, int id);
        //int MC_BarrierInit(MCAgency_t attr, int id, int num_procs);
        //int MC_ChInitializeOptions(MCAgency_t attr, ChOptions_t *options);
        //int MC_CallAgentFunc(MCAgent_t agent, const char* funcName, void* returnVal, void* varg);
        //int MC_CondBroadcast(MCAgency_t attr, int id);
        //int MC_CondSignal(MCAgency_t attr, int id);
        //int MC_CondReset(MCAgency_t attr, int id);
        //int MC_CondWait(MCAgency_t attr, int id);
        //int MC_DeleteAgent(MCAgent_t agent);
        //int MC_End(MCAgency_t attr);
        //MCAgent_t MC_FindAgentByName(MCAgency_t attr, const char *name);
        //MCAgent_t MC_FindAgentByID(MCAgency_t attr, int ID);
        //void* MC_GetAgentExecEngine(MCAgent_t agent);
        //int MC_GetAgentID(MCAgent_t agent);
        //char* MC_GetAgentName(MCAgent_t agent);
        //int MC_GetAgentNumTasks(MCAgent_t agent);
        //int MC_GetAgentReturnData(MCAgent_t agent, int task_num, void **data, int *dim, int **extent);
        //int MC_GetAgentStatus(MCAgent_t agent);
        //enum MC_AgentType_e MC_GetAgentType(MCAgent_t agent);
        //char* MC_GetAgentXMLString(MCAgent_t agent);
        //int MC_HaltAgency(MCAgency_t agency);
        //MCAgency_t MC_Initialize(int port, MCAgencyOptions_t *options);
        //int MC_InitializeAgencyOptions(struct MCAgencyOptions_s* options);
        //int MC_MutexLock(MCAgency_t attr, int id);
        //int MC_MutexUnlock(MCAgency_t attr, int id);
        //int MC_RegisterService(MCAgency_t agency, MCAgent_t agent, int agentID, const char *agentName, char **serviceNames, int numServices);
        //int MC_ResumeAgency(MCAgency_t agency);
        //MCAgent_t MC_RetrieveAgent(MCAgency_t attr);
        //int MC_SemaphorePost(MCAgency_t attr, int id);
        //int MC_SemaphoreWait(MCAgency_t attr, int id);
        //int MC_SetDefaultAgentStatus(MCAgency_t agency, enum MC_AgentStatus_e status);
        //int MC_SetThreadOn(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index);
        //int MC_SetThreadsAllOn(MCAgencyOptions_t* options);
        //int MC_SetThreadOff(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index );
        //int MC_SetThreadsAllOff(MCAgencyOptions_t* options);
        //int MC_PrintAgentCode(MCAgent_t agent);
        //char * MC_RetrieveAgentCode(MCAgent_t agent);
        //int MC_ResetSignal(MCAgency_t attr);
        //int MC_SearchForService(MCAgency_t attr, const char *searchString, char*** agentNames, char*** serviceNames, int** agentIDs, int* numResults);
        //int MC_SendAgentMigrationMessage(MCAgency_t attr, const char *message, const char *hostname, int port);
        //int MC_SendAgentMigrationMessageFile(MCAgency_t attr, const char *filename, const char *hostname, int port);
        //int MC_SetAgentStatus(MCAgent_t agent, int status);
        //int MC_Steer(MCAgency_t attr, int (*funcptr)(void* data), void *arg);
        //enum MC_SteerCommand_e MC_SteerControl(void);
        //int MC_SyncDelete(MCAgency_t attr, int id);
        //int MC_SyncInit(MCAgency_t attr, int id);
        //int MC_TerminateAgent(MCAgent_t agent);
        //int MC_WaitAgent(MCAgency_t attr);
        //MCAgent_t MC_WaitRetrieveAgent(MCAgency_t attr);
        //int MC_WaitSignal(MCAgency_t attr, int signals);
        */
    }
}
