using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

/*!
 * \mainpage LibMC.NET
 * \section intro_sec Introduction
 * Welcome to the LibMC.NET documentation. LibMC.NET is .NET class library
 * that wraps Mobile-C in a layer to make accessible to .NET applications.
 * While the wrapper is continually under development, most of the features
 * of Mobile-C are currently available to the programmer. Please use the links
 * above to find out more.
 *
 * \section compiling Compiling LibMC.NET
 * Compiling: TBD.
 * 
 * \section using Using LibMC.NET
 * Using: TBD.
 *
 * \section examples Example Projects
 * Examples: TBD.
 */

/** \file MCAgency.cs
 * Defines the MCAgency object and its member functions.
 */

/**
 * \example LibMCGui/Program.cs
 * \include LibMCGui/Form1.cs
 * \include LibMCGui/Form1.Designer.cs
 * Basic Mobile-C Windows Forms demo program
 */

/** 
 * \example LibMCConsole/Program.cs
 * Basic Mobile-C console demo program
 */

/** 
 * \example LibMCMiscTest/Program.cs
 * Demonstartes miscellaneous Mobile-C functions.
 */

/**
 * \brief      Namespace for the .NET wrapper for Mobile-C.
 * 
 * LibMC encapsulates the Mobile-C DLL for windows in an .NET class
 * library. .NET programs can access the library to create agencies,
 * connect to agencies, interact with agents, etc.
 */
namespace LibMC
{
    /**
     * \brief   Wrapper class for MCAgency_t structure.
     * 
     * This class provides an interface to the Mobile-C agency. Member
     * functions for the class are generally overloaded versions of the 
     * respective functions in the Mobile-C library. The class maintains a 
     * pointer to the Mobile-C agency in unmanaged memory. The pointer
     * is not accessible by the user.
     */
    public partial class MCAgency
    {
        /**
         * \brief   Enum for describing the state of the agency.
         * 
         * This enum is used to determine whether or not certain actions
         * should be permitted, such as halting, resuming, and ending an agency
         */
        public enum MCAgencyState
        {
            NoState = -1,       /*<! Default, uninitialized state */
            Initialized = 0,    /*<! Agency initialized, but not started */
            Running,            /*<! Agency is running */
            Halted,             /*<! Agency has been stopped (can be resumed) */
            Ended,              /*<! Agency is stopped (destroyed) */
        }

        private IntPtr agency_p = IntPtr.Zero;
        private MCAgencyOptions_t options;
        private int port = -1;
        private MCAgencyState state = MCAgencyState.NoState;

        /** 
         * \brief       Default constructor
         * 
         * The default constructor for the MCAgency class. It creates a new agency,
         * default options for the agency, and initializes the agency. It does not
         * start the agency.
         */
        public MCAgency()
        {
            options = new MCAgencyOptions_t();
            _MC_InitializeAgencyOptions(ref options);
            state = MCAgencyState.Initialized;
        }

        private IntPtr Agency
        {
            get
            {
                if (agency_p == IntPtr.Zero)
                    throw new SystemException("Private agency pointer is zero!");
                else
                    return agency_p;
            }
            set
            {
                if (agency_p == IntPtr.Zero)
                    agency_p = value;
                else
                    throw new SystemException("Attempting to assign new agency to non-zero agency pointer!");
            }
        }

        /**
         * \brief       Accessor for the port number of the agency.
         * 
         * Allows the user to set the agency port or get the port number while it 
         * is running.
         * 
         * \note        The port must be set before the agency is started. Once the agency
         *              is started, the port cannot be changed.
         */
        public int Port
        {
            get
            {
                return port;
            }
            set
            {
                port = value;
            }
        }

        /**
         * \brief       Accessor for the agency state.
         * 
         * Allows the user to query the state of the agency.
         * 
         * \note        The state cannot be set by the user. It is
         *              controlled internally.
         */
        public MCAgencyState State
        {
            get
            {
                return state;
            }
        }

        /*
         * Agency functions
         */

        /**
         * \brief       Starts the agency.
         * 
         * Starts the agency and sets the agency state.
         * 
         * \returns     0 on success, -1 on failure.
         * 
         * \note        The agency port and any other options must be set
         *              before calling this function.
         */
        public int Initialize()
        {
            if (port == -1)
                return -1;
            else
            {
                Agency = MCAgency._MC_Initialize(port, ref options);
                state = MCAgencyState.Running;
                return 0;
            }
        }

        /**
         * \brief       Stops and destroys the agency.
         * 
         * Stops the agency and sets the agency state appropriately.
         * 
         * \returns     The return value of the underlying MC_End function.
         * 
         * \note        This call will fail if the underlying Mobile-C agency
         *              is not in the correct state.
         */
        public int End()
        {
            int temp = MCAgency._MC_End(Agency);
            state = MCAgencyState.Ended;
            return temp;
        }

        /**
         * \brief       Initializes Ch options for the agency.
         * 
         * Can be used to set the home directory and shell mode for the Ch interpretter.
         * 
         * \param       shellType The type of shell Ch should use: CH_REGULARCH or CH_SAFECH.
         * \param       home The home directory Ch should use.
         * \returns     The return value of the underlying MC_ChInitializeOptions function.
         * 
         * \note        This function must be called before the agency is started.
         */
        public int ChInitializeOptions(int shellType, String home)
        {
            ChOptions_t options = new ChOptions_t();
            options.chhome = home;
            options.shelltype = shellType;
            return _MC_ChInitializeOptions(Agency, options);
        }

        /* \brief       Sets all threads for the agency to "on."
         * 
         * Sets all threads for the agency to "on." This is also the default state
         * for the agency.
         * 
         * \returns     The return value of the underlying MC_SetThreadsAllOn function.
         * 
         * \note        This function must be called before the agency is started.
         */
        public int SetThreadsAllOn()
        {
            return _MC_SetThreadsAllOn(ref options);
        }

        /**
         * \brief       Sets all threads for the agency to "off."
         * 
         * Sets all threads for the agency to "off." Not recommended for use.
         * 
         * \returns     The return value of the underlying MC_SetThreadsAllOff function.
         * 
         * \note        This function must be called before the agency is started.
         */
        public int SetThreadsAllOff()
        {
            return _MC_SetThreadsAllOff(ref options);
        }

        /**
         * \brief       Sets an individual thread for the agency to "on."
         * 
         * Threads are on by default. If they have been turned off, this function
         * turns them on again.
         * 
         * \param       index The enum that identifies the thread to be turned on.
         * \returns     The return value of the underlying MC_SetThreadOn function.
         * 
         * \note        This function must be called before the agency is started.
         */
        public int SetThreadOn(MC_ThreadIndex_e index)
        {
            return _MC_SetThreadOn(ref options, index);
        }

        /**
         * \brief       Sets an individual thread for the agency to "off."
         * 
         * Most commonly used to turn the command prompt thread off.
         * 
         * \param       index The enum that identifies the thread to be turned off.
         * \returns     The return value of the underlying MC_SetThreadOff function.
         * 
         * \note        This function must be called before the agency is started.
         */
        public int SetThreadOff(MC_ThreadIndex_e index)
        {
            return _MC_SetThreadOff(ref options, index);
        }

        /**
         * \brief       Temporarily halts the agency.
         * 
         * Halts the agency until it is resumed or ended.
         * 
         * \returns     The return value of the underlying MC_HaltAgency function.
         * 
         * \note        The underlying Mobile-C agency must be in the correct state
         *              to call this function or it will fail.
         */
        public int HaltAgency()
        {
            int temp = _MC_HaltAgency(Agency);
            state = MCAgencyState.Halted;
            return temp;
        }

        /**
         * \brief       Resumes a halted agency.
         * 
         * Resumes a halted agency. Cannot be used on ended agencies.
         * 
         * \returns     The return value of the underlying MC_ResumeAgency function.
         * 
         * \note        The underlying Mobile-C agency must be in the correct state
         *              to call this function or it will fail.
         */
        public int ResumeAgency()
        {
            int temp = _MC_ResumeAgency(Agency);
            state = MCAgencyState.Running;
            return temp;
        }

        /**
         * \brief       Sets the default state of an agent in the agency.
         * 
         * Can be used to set the default status of agents, but most agents
         * managed their state on their own.
         * 
         * \param       status The enum that identifies the desired agent state.
         * \returns     The return value of the underlying MC_SetDefaultAgentStatus function.
         */
        public int SetDefaultAgentStatus(MC_AgentStatus_e status)
        {
            return _MC_SetDefaultAgentStatus(Agency, status);
        }

        /**
         * \brief       Waits for an agent to arrive and returns the agent.
         * 
         * Waits for an agent to arrive in the agency, then returns that agent.
         * The agent is not allowed to execute.
         * 
         * \returns     The agent that was retrieved or an empty agent if it fails.
         */
        public MCAgent WaitRetrieveAgent()
        {
            IntPtr temp = _MC_WaitRetrieveAgent(Agency);
            if (temp != IntPtr.Zero)
                return new MCAgent(temp);
            else
                return new MCAgent();
        }

        /**
         * \brief       Waits for an agent to arrive.
         * 
         * Waits for an agent to arrive in the agency. The agent is 
         * allowed to execute normally.
         * 
         * \returns     The return value of the underlying MC_WaitAgent function.
         */
        public int WaitAgent()
        {
            return _MC_WaitAgent(Agency);
        }

        /*
         * Migration functions
         */

        /**
         * \brief       Sends an agent migration message file to an agency.
         * 
         * Sends the specified XML file to another agency (local or remote).
         * 
         * \param       filename The name of the file to send (fully qualified).
         * \param       hostname The URL, IP address, or other identifier for the agency host.
         * \param       port The port to send to.
         * \returns     The return value of the underlying MC_SendAgentMigrationMessageFile function.
         */
        public int SendAgentMigrationMessageFile(String filename, String hostname, int port)
        {
            return MCAgency._MC_SendAgentMigrationMessageFile(Agency, filename, hostname, port);
        }

        /**
         * \brief       Sends an agent migration message to an agency.
         * 
         * Sends an agent migration message to another agency (local or remote).
         * 
         * \param       message The agent migration message.
         * \param       hostname The URL, IP address, or other identifier for the agency host.
         * \param       port The port to send to.
         * \returns     The return value of the underlying MC_SendAgentMigrationMessageFile function.
         */
        public int SendAgentMigrationMessage(String message, String hostname, int port)
        {
            return _MC_SendAgentMigrationMessage(Agency, message, hostname, port);
        }

        /*
         * Signals and threading functions
         */

        /**
         * \brief       Broadcast a condition signal.
         * 
         * Broadcasts a signal in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit().
         * 
         * \param       id The ID number of the condition to signal.
         * \returns     The return value of the underlying MC_CondBroadcast function.
         */
        public int CondBroadcast(int id)
        {
            return _MC_CondBroadcast(Agency, id);
        }

        /**
         * \brief       Signal a condition.
         * 
         * Signals a condition in the agency. The parameter "id" is the ID of
         * the agency sync variable to signal that was created with SyncInit().
         * 
         * \param       id The ID number of the condition to signal.
         * \returns     The return value of the underlying MC_CondSignal function.
         */
        public int CondSignal(int id)
        {
            return _MC_CondSignal(Agency, id);
        }

        /**
         * \brief       Reset a condition signal.
         * 
         * Resets a signal in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit(). This function
         * must be called after a condition is received in order to clear it.
         * 
         * \param       id The ID number of the condition to reset.
         * \returns     The return value of the underlying MC_CondReset function.
         */
        public int CondReset(int id)
        {
            return _MC_CondReset(Agency, id);
        }

        /**
         * \brief       Wait for a condition signal.
         * 
         * Waits for a condition signal in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit(). This function blocks
         * until the signal is received.
         * 
         * \param       id The ID number of the condition to wait for.
         * \returns     The return value of the underlying MC_CondWait function.
         */
        public int CondWait(int id)
        {
            return _MC_CondWait(Agency, id);
        }

        /**
         * \brief       Lock a mutex.
         * 
         * Locks a mutex in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit(). This function
         * blocks until the mutex is locked.
         * 
         * \param       id The ID number of the mutex to lock.
         * \returns     The return value of the underlying MC_MutexLock function.
         */
        public int MutexLock(int id)
        {
            return _MC_MutexLock(Agency, id);
        }

        /**
         * \brief       Unlock a mutex.
         * 
         * Locks a mutex in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit().
         * 
         * \param       id The ID number of the mutex to unlock.
         * \returns     The return value of the underlying MC_MutexUnlock function.
         */
        public int MutexUnlock(int id)
        {
            return _MC_MutexUnlock(Agency, id);
        }

        /**
         * \brief       Posts a semaphore.
         * 
         * Posts a sempaphore in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit().
         * 
         * \param       id The ID number of the semaphore to post.
         * \returns     The return value of the underlying MC_SemaphorePost function.
         */
        public int SemaphorePost(int id)
        {
            return _MC_SemaphorePost(Agency, id);
        }

        /**
         * \brief       Wait for a semaphore to be posted.
         * 
         * Wait for a semaphore in the agency to be posted. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit(). This function blocks until
         * the semaphore is posted.
         * 
         * \param       id The ID number of the semaphore to wait for.
         * \returns     The return value of the underlying MC_SemaphoreWait function.
         */
        public int SemaphoreWait(int id)
        {
            return _MC_SemaphoreWait(Agency, id);
        }

        /**
         * \brief       Reset an agency signal.
         * 
         * Resets a signal in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit().
         * 
         * \returns     The return value of the underlying MC_ResetSignal function.
         */
        public int ResetSignal()
        {
            return _MC_ResetSignal(Agency);
        }

        /**
         * \brief       Delete a synchronization variable.
         * 
         * Deletes a synchronization variable in the agency. The parameter "id" is the ID of
         * the agency sync variable created with SyncInit().
         * 
         * \param       id The ID number of the variable to delete.
         * \returns     The return value of the underlying MC_SyncDelete function.
         */
        public int SyncDelete(int id)
        {
            return _MC_SyncDelete(Agency, id);
        }

        /**
         * \brief       Create a new synchronization variable.
         * 
         * Creates a new synchronization variable in the agency. The parameter "id" is 
         * desired ID of the variable. A random ID is returned if "id" is already in use.
         * 
         * \param       id The ID number of the condition to signal.
         * \returns     The return value of the underlying MC_CondBroadcast function-
         *              either a random ID or the desired ID if the desired ID is already
         *              in use.
         */
        public int SyncInit(int id)
        {
            return _MC_SyncInit(Agency, id);
        }

        /**
         * \brief       Wait for agency signals.
         * 
         * Waits for signals to occur in the agency.
         * 
         * \param       signals The ID number of the condition to signal.
         * \returns     The return value of the underlying MC_WaitSignal function.
         */
        public int WaitSignal(int signals)
        {
            return _MC_WaitSignal(Agency, signals);
        }

        /**
         * \brief       Delete a barrier object.
         * 
         * Deletes a barrier object from the agency. The parameter "id" is the ID of
         * the agency sync variable created with BarrierInit().
         * 
         * \param       id The ID number of the barrier to delete.
         * \returns     The return value of the underlying MC_BarrierDelete function.
         */
        public int BarrierDelete(int id)
        {
            return _MC_BarrierDelete(Agency, id);
        }

        /**
         * \brief       Create a new barrier.
         * 
         * Creates a new barrier object in the agency.
         * 
         * \param       id The ID number of the condition to signal.
         * \param       num_procs the number of process to block (?)
         * \returns     The return value of the underlying MC_BarrierInit function.
         */
        public int BarrierInit(int id, int num_procs)
        {
            return _MC_BarrierInit(Agency, id, num_procs);
        }

        /*
         * Steering functions
         */

        /**
         * \brief       Steering control function.
         * 
         * Really not sure.
         * 
         * \returns     The return value of the underlying MC_CondBroadcast function.
         */
        public MC_SteerCommand_e SteerControl()
        {
            return _MC_SteerControl();
        }

        /*public int _MC_Steer(IntPtr agency, int (*funcptr)(void* data), void *arg);*/

        /*
         * Services
         */

        /**
         * \brief       Registers services in the agency.
         * 
         * Registers services provided by agents with the agency. Not really useful
         * in binary space.
         * 
         * \param       agent The agent providing the services.
         * \param       agentID The agent ID number.
         * \param       agentName The agent name.
         * \param       serviceNames An array of service names.
         * \param       numServices The number of services provided.
         * \returns     The return value of the underlying MC_RegisterService function.
         */
        public int RegisterService(MCAgent agent, int agentID, String agentName, String[] serviceNames, int numServices)
        {
            return _MC_RegisterService(Agency, agent.Agent, agentID, agentName, serviceNames, numServices);
        }

        /*public int _MC_SearchForService(IntPtr agency, String searchString, char*** agentNames, char*** serviceNames, int** agentIDs, int* numResults);*/

        /*
         * Agent functions
        */

        /**
         * \brief       Add an agent to the agency.
         * 
         * Adds an agent to the agency.
         * 
         * \param       agent The agent to add.
         * \returns     The return value of the underlying MC_AddAgent function.
         */
        public int AddAgent(MCAgent agent)
        {
            return _MC_AddAgent(Agency, agent.Agent);
        }

        /**
         * \brief       Finds an agent by its name.
         * 
         * Finds an agent in the agency by its name.
         * 
         * \param       name The name of the agent to search for.
         * \returns     The return value of the underlying MC_FindAgentByName function.
         */
        public MCAgent FindAgentByName(String name)
        {
            return new MCAgent(_MC_FindAgentByName(Agency, name));
        }

        /**
         * \brief       Find an agent by its ID.
         * 
         * Finds an agent in the agency by its ID number.
         * 
         * \param       id The ID number of the condition to signal.
         * \returns     The return value of the underlying MC_FindAgentByID function.
         */
        public MCAgent FindAgentByID(int id)
        {
            return new MCAgent(_MC_FindAgentByID(Agency, id));
        }

        /**
         * \brief       Retrieve an agent from the agency.
         * 
         * Really not sure.
         * 
         * \returns     The return value of the underlying MC_CondBroadcast function.
         */
        public MCAgent RetrieveAgent()
        {
            return new MCAgent(_MC_RetrieveAgent(Agency));
        }

        /*
         * ACL Functions
         */

        /**
         * \brief       Send an ACL message to the agency.
         * 
         * Sends an ACL message to the agency. The message is delivered appropriately.
         * 
         * \param       acl_message The message to send.
         * \returns     The return value of the underlying MC_CondBroadcast function.
         */
        public int AclSend(MCAclMessage acl_message)
        {
            return MCAgency._MC_AclSend(Agency, acl_message.AclMsg);
        }
    }
}
