using System;
using System.Collections.Generic;
using System.Text;

/** \file MCAclMessage.cs
 * Defines the MCAclMessage object and its member functions.
 */

/** \example LibMCFipaTest/Program.cs
 * Mobile-C FIPA ACL message demo program.
 */

namespace LibMC
{
    /**
     * \brief       Encapsulates ACL messages in the Mobile-C library.
     * 
     * This class contains a pointer to an ACL message in the Mobile-C library.
     * Functions are provided to send the message, set its various fields, and
     * destory the message.
     */
    public class MCAclMessage
    {
        private IntPtr aclmsg_p;

        /**
         * \brief       Default constructor.
         * 
         * Creates an empty ACL message object.
         */
        public MCAclMessage()
        {
            // NOT using accessor!
            aclmsg_p = IntPtr.Zero;
        }

        internal MCAclMessage(IntPtr msg)
        {
            AclMsg = msg;
        }
        
        /*
         * Accessors
         */
        internal IntPtr AclMsg
        {
            get
            {
                if (aclmsg_p == IntPtr.Zero)
                    throw new SystemException("Private acl message pointer is zero!");
                else
                    return aclmsg_p;
            }
            set
            {
                if (aclmsg_p == IntPtr.Zero)
                {
                    aclmsg_p = value;
                }
                else
                    throw new SystemException("Attempting to assign new acl message to non-zero acl message pointer!");
            }
        }

        /**
         * \brief       Creates a new, blank ACL message.
         * 
         * Creates a new ACL message. The message is blank but valid.
         */
        public void New()
        {
            AclMsg = MCAgency._MC_AclNew();
        }

        /**
         * \brief       Creates an ACL message that is a response to the argument.
         * 
         * Creates an ACL message to respond to the argument.
         * 
         * \param       acl_message The message from which to create the reply.
         * \returns     A new ACL message that is a response to the argument or
         *              an empty message if there is an error.
         */
        public MCAclMessage Reply(MCAclMessage acl_message)
        {
            IntPtr temp = MCAgency._MC_AclReply(acl_message.AclMsg);
            if (temp == IntPtr.Zero)
                return new MCAclMessage();
            else
                return new MCAclMessage(temp);
        }

        /**
         * \brief       Sets the performative field of the message.
         * 
         * Sets the performative field of the message.
         * 
         * \param       performative The fipa_performative_e enum describing the message.
         * \returns     The return value of the underlying MC_AclSetPerformative function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetPerformative(MC_FipaPerformative_e performative)
        {
            return MCAgency._MC_AclSetPerformative(AclMsg, performative);
        }

        /**
         * \brief       Sets the sender field of the message.
         * 
         * Sets the performative field of the message.
         * 
         * \param       name The name of the sending entity.
         * \param       address The address of the sending entity.
         * \returns     The return value of the underlying MC_AclSetSender function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetSender(String name, String address)
        {
            return MCAgency._MC_AclSetSender(AclMsg, name, address);
        }

        /**
         * \brief       Adds a receiver to the list of receivers.
         * 
         * Adds a receiver to the list of receivers for the message.
         * 
         * \param       name The name of the receiver.
         * \param       address The address of the receiver.
         * \returns     The return value of the underlying MC_AclAddReceiver function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int AddReceiver(String name, String address)
        {
            return MCAgency._MC_AclAddReceiver(AclMsg, name, address);
        }

        /**
         * \brief       Adds a "reply-to" field to the message.
         * 
         * Adds a "reply-to" field to the message. The reply-to field overrides
         * the sender field when creating a reply.
         * 
         * \param       name The name of the receiver.
         * \param       address The address of the receiver.
         * \returns     The return value of the underlying MC_AclAddAddReplyTo function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int AddReplyTo(String name, String address)
        {
            return MCAgency._MC_AclAddReplyTo(AclMsg, name, address);
        }

        /**
         * \brief       Sets the content field of the message.
         * 
         * Sets the content field of the message.
         * 
         * \param       content The string to copy to the content field.
         * \returns     The return value of the underlying MC_AclSetContent function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetContent(String content)
        {
            return MCAgency._MC_AclSetContent(AclMsg, content);
        }

        /**
         * \brief       Destroys a message.
         * 
         * This function destroys a message in the Mobile-C library.
         * It releases the underlying memory and must be called when the message
         * is no longer needed. 
         * 
         * \returns     The return value of the underlying MC_AclDestroy function.
         * 
         * \note        The message must be a valid message or this function will fail.
         *              In addition, messages are not automatically destroyed by the 
         *              garbage collector. Use care when creating messages and ensure
         *              they are properly destroyed.
         */
        public int Destroy()
        {
            int temp = MCAgency._MC_AclDestroy(AclMsg);
            if (temp == 0)
                aclmsg_p = IntPtr.Zero;
            return temp;
        }

    }
}
