/*[
 * Copyright (c) 2007 Integration Engineering Laboratory
                      University of California, Davis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the complete modified source code.  Modifications are to
 * be distributed as patches to the released version.  Permission to
 * distribute binaries produced by compiling modified sources is granted,
 * provided you
 *   1. distribute the corresponding source modifications from the
 *    released version in the form of a patch file along with the binaries,
 *   2. add special version identification to distinguish your version
 *    in addition to the base release version number,
 *   3. provide your name and address as the primary contact for the
 *    support of your modified version, and
 *   4. retain our contact information in regard to use of the base
 *    software.
 * Permission to distribute the released version of the source code along
 * with corresponding source modifications in the form of a patch file is
 * granted with same provisions 2 through 4 for binary distributions.
 *
 * This software is provided "as is" without express or implied warranty
 * to the extent permitted by applicable law.
]*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef _WIN32
#include "config.h"
#else
#include "winconfig.h"
#endif
#include "include/connection.h"
#include "include/mtp_http.h"
#include "include/macros.h"
#include "include/mc_error.h"
#include "include/message.h"

#define SOCKET_INPUT_SIZE 4096
int
mtp_http_Destroy(mtp_http_p http)
{
#define SAFE_FREE(elem) \
  if(elem) \
    free(elem)

  SAFE_FREE(http->http_version);
  SAFE_FREE(http->host);
  SAFE_FREE(http->return_code);
  SAFE_FREE(http->target);
  SAFE_FREE(http->date);
  SAFE_FREE(http->server);
  SAFE_FREE(http->accept_ranges);
  SAFE_FREE(http->content_length);
  SAFE_FREE(http->connection);
  SAFE_FREE(http->content_type);
#undef SAFE_FREE
  return 0;
}
mtp_http_p
mtp_http_New(void)
{
  mtp_http_p http;
  http = (mtp_http_p)malloc(sizeof(mtp_http_t));
  CHECK_NULL(http, exit(0););
  memset(http, 0, sizeof(mtp_http_t));
  return http;
}

int
mtp_http_InitializeFromConnection
(
 mtp_http_p http,
 connection_p connection
 )
{
  int i=1;
  int n;
  char *message_string;
  char *buffer;
  int message_size = 0;
  int expected_content_size = -1;
  int content_size = 0;

  buffer = (char*) malloc(sizeof(char) * (SOCKET_INPUT_SIZE + 1));
  CHECK_NULL(buffer, exit(0););
  message_string = (char*) malloc(sizeof(char) * (SOCKET_INPUT_SIZE + 1));
  CHECK_NULL(message_string, exit(0););
  message_string[0] = '\0';
  buffer[0] = '\0';

  while(1){
#ifndef _WIN32
    n = recvfrom(connection->clientfd,
        (void *) buffer,
        (size_t) sizeof(char)*SOCKET_INPUT_SIZE,
        0,
        (struct sockaddr *) 0,
        (socklen_t *) 0);
#else
    n = recvfrom(connection->clientfd,
        (void *) buffer,
        (size_t) sizeof(char)*SOCKET_INPUT_SIZE,
        0,
        (struct sockaddr *) 0,
        0);
#endif
    if (n < 0) {
      free(buffer);
      return MC_ERR_CONNECT;
    } 
    else if (n == 0) {
      free(buffer);
      break;
    } else {
      message_size += n;
      buffer[n] = '\0';
      i++;
      strcat(message_string, buffer);
      message_string = realloc
        (
         message_string, 
         sizeof(char) * (SOCKET_INPUT_SIZE+1) * i
        );
      CHECK_NULL(message_string, exit(0););
      buffer[0] = '\0';
      if (!strcmp
          (
           message_string + message_size - 4,
           "\r\n\r\n"
          )
         )
        break;
    }
  }
  
  if(mtp_http_Parse(http, message_string)) {
    /* Reply with an HTTP error code */
    buffer = malloc
      (
       sizeof(char) * 
       (
        strlen
        (
         "HTTP/1.0 503 Service Unavailable\r\nConnection: Close\r\n\r\nMobile C"
        )+1
       )
      );
    strcpy
      (
       buffer,
       "HTTP/1.0 503 Service Unavailable\r\nConnection: Close\r\n\r\nMobile C"
      );
    send
      (
       connection->clientfd,
       (void*)buffer,
       sizeof(char)*(strlen(buffer)),
       0
      );
    free(message_string);
    return ERR;
  } else {
    free(message_string);
    return 0;
  }
}

const char* http_GetExpression(const char* string, char** expr)
{
  int i;
  int j;
  int success_flag = 0;
  const char* next_expr_ptr;
  for(i = 0;string[i] != '\0';i++) {
    if (
        (
         (string[i] == '\r') &&
         (string[i+1] == '\n')
        ) 
        ||
        (string[i] == '\n')
       )
    {
      success_flag = 1;
      break;
    } 
  }
  if(success_flag)
  {
    *expr = (char*)malloc
      (
       sizeof(char) * (i+1)
      );
    for(j = 0; j < i; j++) {
      (*expr)[j] = string[j];
    }
    (*expr)[j] = '\0';
    next_expr_ptr = &(string[i]);
    while
      (
       (*next_expr_ptr == '\r') ||
       (*next_expr_ptr == '\n')
      )
      {
        next_expr_ptr++;
      }
    return next_expr_ptr;
  } else {
    return NULL;
  }
}
      
int http_ParseExpression(
    const char* expression_string,
    char** name,
    char** value
    )
{
  int i=0;
  const char* tmp;
  const char* charptr;
  tmp = expression_string;
  if (tmp == NULL) {
    return MC_ERR_PARSE;
  }
  for(; *tmp!=':' && *tmp!='\0'; tmp++)
    i++;
  if(*tmp == '\0') {
    *name = NULL;
    *value = NULL;
    return MC_ERR_PARSE;
  }
  *name = (char*)malloc
    (
     sizeof(char) * (i+1)
    );
  CHECK_NULL(*name, exit(0););
  charptr = expression_string;
  i=0;
  while(charptr != tmp) {
    (*name)[i] = *charptr;
    i++;
    charptr++;
  }
  (*name)[i] = '\0';

  tmp++;
  while
    (
     (*tmp == ' ') ||
     (*tmp == '\t')
    )
      tmp++;

  *value = (char*)malloc
    (
     sizeof(char) * 
     (strlen(tmp) + 1)
    );
  CHECK_NULL(*value, exit(0););
  strcpy(*value, tmp);
  return MC_SUCCESS;
}

int
mtp_http_Parse(struct mtp_http_s* http, const char* string)
{
  const char* line = NULL;
  char* expr = NULL;
  char* name = NULL;
  char* value = NULL;

  int err_code = 0;

  line = string;
  line = http_ParseHeader
    (
     http,
     line
    );
  do
  {
    line = http_GetExpression
      (
       line,
       &expr
      );

    err_code = http_ParseExpression
      (
       expr,
       &name,
       &value
      );
    if
      (
       (name == NULL) ||
       (value == NULL)
      )
      {
        if (expr != NULL) {
          free(expr);
        }
        break;
      }
#define HTTP_PARSE_EXPR( parse_name, struct_name ) \
    if ( !strcmp(name, parse_name) ) { \
      http->struct_name = (char*)malloc \
      ( \
        sizeof(char) * \
        (strlen(value)+1) \
      ); \
      strcpy(http->struct_name, value); \
    } else

    HTTP_PARSE_EXPR( "Host", host )
    HTTP_PARSE_EXPR( "Date", date )
    HTTP_PARSE_EXPR( "Server", server )
    HTTP_PARSE_EXPR( "Accept-Ranges", accept_ranges )
    HTTP_PARSE_EXPR( "Content-Length", content_length)
    HTTP_PARSE_EXPR( "Connection", connection )
    HTTP_PARSE_EXPR( "Content-Type", content_type)
    HTTP_PARSE_EXPR( "User-Agent", user_agent)
    {
      fprintf(stderr, "Warning: Unknown http name: %s. %s:%d\n",
          name, __FILE__, __LINE__);
    }
#undef HTTP_PARSE_EXPR

#define SAFE_FREE( object ) \
    if(object) free(object); \
    object = NULL

    SAFE_FREE(expr);
    SAFE_FREE(name);
    SAFE_FREE(value);
#undef SAFE_FREE
      
  } while(line != NULL && err_code == MC_SUCCESS);
  /* Copy rest of contents into the data member */
  if (line != NULL) {
    http->data = (void*)malloc
      (
       sizeof(char) * 
       (strlen(line)+1)
      );
    strcpy((char*)http->data, line);
  }
  if (
      (http->http_performative == HTTP_POST) ||
      (http->http_performative == HTTP_PUT)
     )
    return 0;
  else
    return 1;
}

const char* http_ParseHeader(
    mtp_http_p http,
    const char* string )
{
  const char* cur;
  char* token;
  char* tmp = NULL;

  cur = string;
  cur = http_GetToken(cur, &token);
  if (token == NULL) {
    return NULL;
  }
  if (!strcmp(token, "GET")) {
    http->http_performative = HTTP_GET;
    cur = http_GetToken(cur, &tmp);
    /* We don't support 'get' yet */
    if(tmp) free(tmp);
  } else if(!strcmp(token, "HEAD")) {
    /* Don't support this yet */
    http->http_performative = HTTP_HEAD;
  } else if(!strcmp(token, "POST")) {
    http->http_performative = HTTP_POST;
    cur = http_GetToken(cur, &tmp);
    if(tmp != NULL) {
      http->target = (char*) malloc(sizeof(char) * (strlen(tmp)+1));
      strcpy(http->target, tmp);
      free(tmp);
    }
  } else if(!strcmp(token, "PUT")) {
    http->http_performative = HTTP_PUT;
    cur = http_GetToken(cur, &tmp);
    if (tmp != NULL) {
      http->target = (char*)malloc(sizeof(char)*(strlen(tmp)+1));
      strcpy(http->target, tmp);
      free(tmp);
    }
  } else if(!strcmp(token, "DELETE")) {
    http->http_performative = HTTP_DELETE;
  } else if(!strcmp(token, "TRACE")) {
    http->http_performative = HTTP_TRACE;
  } else if(!strcmp(token, "OPTIONS")) {
    http->http_performative = HTTP_OPTIONS;
  } else if(!strcmp(token, "CONNECT")) {
    http->http_performative = HTTP_CONNECT;
    /* FIXME */
  } else {
    /* Illegal performative */
    http->http_performative = HTTP_PERFORMATIVE_UNDEF;
  }
  free(token);
  cur = string;
  while(*cur != '\0') {
    if(*cur == '\n') {
      while (*cur == '\n' || *cur == '\r' || *cur == ' ')
        cur++;
      break;
    }
    cur++;
  }
  return cur;
}

const char*
http_GetToken(const char* string, char** token)
{
  const char* cur;
  const char* begin;
  char* itr;
  
  cur = string;
  /* See if it's an empty line */
  if (string[0] == '\r' && string[1] == '\n') {
    *token = NULL;
    return NULL;
  }
  /* Get rid of initial whitespace */
  while(*cur == ' ' || *cur == '\t' || *cur == '\r' || *cur == '\n') cur++;

  begin = cur;
  while(*cur != '\0') {
    cur++;
    if (*cur == ' ' || *cur == '\t' || *cur == '\r' || *cur == '\n')
      break;
  }
  cur--;
  *token = (char*)malloc(cur - begin + 2*sizeof(char));
  itr = *token;
  while (begin <= cur) {
    *itr = *begin;
    itr++;
    begin++;
  }
  *itr='\0';
  return cur+1;
}

int 
mtp_http_ComposeMessage(message_p message)
{
  char* http_header;
  char* tmp;
  char buf[20];

  http_header = (char*)malloc
    (
     sizeof(char) * (1400 + strlen(message->to_address))
    );
  http_header[0] = '\0';
  strcat(http_header, "POST /ams HTTP/1.0\r\n");
  strcat(http_header, "User-Agent: MobileC/");
  strcat(http_header, PACKAGE_VERSION);
  strcat(http_header, "\r\n");
  strcat(http_header, "Host: ");
  strcat(http_header, message->to_address);
  strcat(http_header, "\r\n");
  strcat(http_header, "Content-Type: text/plain\r\n");
  strcat(http_header, "Connection: Close\r\n");
  strcat(http_header, "Content-Length: ");
  sprintf(buf, "%d", strlen(message->message_body) + 1);
  strcat(http_header, buf);
  strcat(http_header, "\r\n\r\n");

  tmp = (char*)malloc
    (
     sizeof(char) * 
     (strlen(http_header) + strlen(message->message_body) + 1)
    );
  tmp[0] = '\0';
  strcpy(tmp, http_header);
  strcat(tmp, message->message_body);
  free(message->message_body);
  message->message_body = tmp;
  free(http_header);
  return MC_SUCCESS;
}
