/*[
 * Copyright (c) 2007 Integration Engineering Laboratory
                      University of California, Davis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the complete modified source code.  Modifications are to
 * be distributed as patches to the released version.  Permission to
 * distribute binaries produced by compiling modified sources is granted,
 * provided you
 *   1. distribute the corresponding source modifications from the
 *    released version in the form of a patch file along with the binaries,
 *   2. add special version identification to distinguish your version
 *    in addition to the base release version number,
 *   3. provide your name and address as the primary contact for the
 *    support of your modified version, and
 *   4. retain our contact information in regard to use of the base
 *    software.
 * Permission to distribute the released version of the source code along
 * with corresponding source modifications in the form of a patch file is
 * granted with same provisions 2 through 4 for binary distributions.
 *
 * This software is provided "as is" without express or implied warranty
 * to the extent permitted by applicable law.
]*/

#ifndef _WIN32
#include <unistd.h>
#include <pthread.h>
#else
#include <windows.h>
#endif
#include <embedch.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "include/libmc.h"
#include "include/macros.h"
#include "include/mc_platform.h"
#include "include/message.h"
#include "include/data_structures.h"
#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX 255
#endif

/* The Global platform variable */
mc_platform_p g_mc_platform;

/* **************************** *
 * Libmc Binary Space Functions *
 * **************************** */
int 
MC_AddAgent(MCAgency_t attr, MCAgent_t agent) /*{{{*/
{
  agent->mc_platform = attr->mc_platform;

  agent_queue_Add(attr->mc_platform->agent_queue, agent);

  MUTEX_LOCK(attr->mc_platform->ams->runflag_lock);
  attr->mc_platform->ams->run = 1;
  COND_SIGNAL(attr->mc_platform->ams->runflag_cond);
  MUTEX_UNLOCK(attr->mc_platform->ams->runflag_lock);
  return 0;
} /*}}}*/

int 
MC_Barrier(MCAgency_t attr, int id) /*{{{*/
{
    barrier_queue_p list = attr->mc_platform->barrier_queue;
    barrier_node_p node;
    node = barrier_queue_Get(list, id);
    if(node == NULL) {
        return MC_ERR_NOT_FOUND;
    }

    MUTEX_LOCK(node->lock);
    node->num_waiting++;
    if (node->num_waiting >= node->num_registered) {
        /* Wake all agents/threads waiting on this barrier */
        COND_BROADCAST(node->cond);
        MUTEX_UNLOCK(node->lock);
        return MC_SUCCESS;
    } else {
        while (node->num_waiting < node->num_registered) {
            COND_WAIT(node->cond, node->lock);
        }
        MUTEX_UNLOCK(node->lock);
    }
    return MC_SUCCESS;
} /*}}}*/

int
MC_BarrierInit(MCAgency_t attr, int id, int num_procs) /*{{{*/
{
    barrier_node_p node;
    /* First see if there already exists a barrier of the same ID. */
    node = barrier_queue_Get(attr->mc_platform->barrier_queue, id);
    if (node != NULL) {
      return MC_ERR;
    }
    node = barrier_node_Initialize(id, num_procs);
    barrier_queue_Add(attr->mc_platform->barrier_queue, node);
    return MC_SUCCESS;
} /*}}}*/

int 
MC_BarrierDelete(MCAgency_t attr, int id) /*{{{*/
{
  return barrier_queue_Delete(id, attr->mc_platform->barrier_queue);
} /*}}}*/

int 
MC_CallAgentFunc( /*{{{*/
        MCAgent_t agent,
        const char* funcName,
        void* returnVal,
        void* varg)
{
    int return_code;
    MUTEX_LOCK(agent->run_lock);
    return_code = Ch_CallFuncByName(
            agent->agent_interp,
            funcName,
            returnVal,
            varg);
    MUTEX_UNLOCK(agent->run_lock);
    return return_code;
} /*}}}*/ 

int 
MC_ChInitializeOptions(MCAgency_t attr, ChOptions_t *options) { /*{{{*/
    if(attr->mc_platform == NULL) {
        fprintf(stderr, "MC_ChInitializeOptions must be called after MC_Start()\n");
        fprintf(stderr, "Using default interpretor options...\n");
        return 1;
    }
    else {
        if (attr->mc_platform->interp_options == NULL) {
          attr->mc_platform->interp_options = (ChOptions_t*)malloc(
              sizeof(ChOptions_t) );
        }
        *attr->mc_platform->interp_options = *options;
        attr->mc_platform->interp_options->chhome = strdup(options->chhome);
        return 0;
    }
} /*}}}*/

int 
MC_CondBroadcast(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    condnode->signalled=1;
    COND_BROADCAST(condnode->cond);
    MUTEX_UNLOCK(condnode->lock);
    return 0;
} /*}}}*/

int 
MC_CondSignal(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    condnode->signalled=1;
    COND_SIGNAL(condnode->cond);
    MUTEX_UNLOCK(condnode->lock);
    return 0;
} /*}}}*/

int 
MC_CondWait(MCAgency_t attr, int id)  /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    if (condnode->signalled) {
        MUTEX_UNLOCK(condnode->lock);
        return 1;
    }

    while (condnode->signalled == 0) {
      COND_WAIT(condnode->cond, condnode->lock);
    }
    MUTEX_UNLOCK(condnode->lock);

    return 0;
} /*}}}*/

int 
MC_CondReset(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    if (condnode->signalled) {
        condnode->signalled = 0;
        MUTEX_UNLOCK(condnode->lock);
        return 0;
    }
    MUTEX_UNLOCK(condnode->lock);
    return 1;
} /*}}}*/

int 
MC_CopyAgent(MCAgent_t* agent_out, const MCAgent_t agent_in) /*{{{*/
{
  *agent_out = agent_Copy(agent_in);
  return MC_SUCCESS;
} /*}}}*/

int 
MC_DeleteAgent(MCAgent_t agent) /*{{{*/
{
    /* Error Checking */
    CHECK_NULL(agent, return MC_ERR_INVALID;);

    /* First, make sure the agent is no longer running */
    MC_TerminateAgent(agent);

    /* Now, we just set it's status to MC_WAIT_FINISHED so that the AMS will
     * flush it next chance it gets. */
    MC_SetAgentStatus(agent, MC_WAIT_FINISHED);
    return MC_SUCCESS;
} /*}}}*/

int 
MC_DeregisterService( /*{{{*/
        MCAgency_t agency,
        int agentID,
        const char *serviceName)
{
    int err_code;

    df_request_list_node_t *req_node;
    df_deregister_p deregister_data;

    req_node = df_request_list_node_New();
    req_node->command = (char*)malloc(sizeof(char)*11);

    strcpy((char*)req_node->command, "deregister");

    deregister_data = (df_deregister_p)malloc(sizeof(df_deregister_t));
    deregister_data->agent_id = agentID;
    deregister_data->service_name = (char*)serviceName;

    req_node->data = deregister_data;

    err_code = df_AddRequest(
            agency->mc_platform->df,
            req_node
            );
    return err_code;
} /*}}}*/

int
MC_End(MCAgency_t agency) /*{{{*/
{
  /* Now, we must stop all the running pthreads somehow... */
  /* We will set the quit flag and signal some important threads to make
   * sure they exit cleanly. We want the df, ams, and especially the acc
   * to be done with whatever they are doing before exiting. */
  MUTEX_LOCK(agency->mc_platform->quit_lock);
  agency->mc_platform->quit = 1;
  MUTEX_UNLOCK(agency->mc_platform->quit_lock);
  COND_SIGNAL(agency->mc_platform->df->request_list->cond);
  COND_SIGNAL(agency->mc_platform->ams->runflag_cond);
  COND_SIGNAL(agency->mc_platform->connection_queue->cond);
  COND_SIGNAL(agency->mc_platform->message_queue->cond);
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_ACC)) {
    THREAD_CANCEL( agency->mc_platform->acc->listen_thread );
  }
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_CP)) {
    THREAD_CANCEL( agency->mc_platform->cmd_prompt->thread );
  }

  if( GET_THREAD_MODE( agency->threads, MC_THREAD_DF)) {
    THREAD_JOIN(agency->mc_platform->df->thread);
  }
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_AMS)) {
    THREAD_JOIN(agency->mc_platform->ams->thread);
  }
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_ACC)) {
    THREAD_JOIN(agency->mc_platform->acc->thread);
    THREAD_JOIN(agency->mc_platform->acc->message_handler_thread);
  }

  mc_platform_Destroy(agency->mc_platform);

  if (agency->hostName)
    free(agency->hostName); 
  free(agency);

  return 0;
} /*}}}*/

MCAgent_t 
MC_FindAgentByName( MCAgency_t attr, /*{{{*/
    const char *name) 
{
  extern mc_platform_p g_mc_platform;
  if (attr == NULL) {
    return agent_queue_SearchName(g_mc_platform->agent_queue, name);
  } else {
    return agent_queue_SearchName(attr->mc_platform->agent_queue,
        name);
  }
} /*}}}*/

MCAgent_t
MC_FindAgentByID( MCAgency_t attr, /*{{{*/
    int ID)
{
  extern mc_platform_p g_mc_platform;
  if (attr == NULL) {
    return agent_queue_Search(g_mc_platform->agent_queue, ID);
  } else {
    return agent_queue_Search(attr->mc_platform->agent_queue,
        ID);
  }
} /*}}}*/

#ifndef _WIN32
  time_t 
#else
  SYSTEMTIME
#endif
MC_GetAgentArrivalTime(MCAgent_t agent) /*{{{*/
{
  if (agent != NULL) {
    return agent->arrival_time;
  } else {
#ifndef _WIN32
    return (time_t)-1;
#else
    SYSTEMTIME oy;
    return oy;
#endif
  }
} /*}}}*/

int 
MC_GetAgentStatus(MCAgent_t agent) /*{{{*/
{
  int status;
  MUTEX_LOCK(agent->lock);
  status = agent->agent_status;
  MUTEX_UNLOCK(agent->lock);
  return status;
} /*}}}*/

char*
MC_GetAgentXMLString(MCAgent_t agent) /*{{{*/
{
  char *ret;
  ret = mxmlSaveAllocString(
      agent->datastate->xml_agent_root,
      NULL
      );
  return ret;
} /*}}}*/

/*ChInterp_t */
void*
MC_GetAgentExecEngine(MCAgent_t agent) /*{{{*/
{
  return agent->agent_interp;
} /*}}}*/

int 
MC_GetAgentID( /*{{{*/
    MCAgent_t agent
    )
{
  return agent->id;
} /*}}}*/

char* 
MC_GetAgentName( /*{{{*/
    MCAgent_t agent
    )
{
  char *name;
  MUTEX_LOCK(agent->lock);
  name = (char*)malloc(sizeof(char) * 
      (strlen (agent->name) + 1)
      );
  strcpy(
      name,
      agent->name
      );
  MUTEX_UNLOCK(agent->lock);
  return name;
} /*}}}*/

int
MC_GetAgentReturnData( /*{{{*/
    MCAgent_t agent,
    int task_num,
    void **data,
    int *dim,
    int **extent) 
{
  int num_elements;
  int size;
  int i;
  if (task_num >= agent->datastate->number_of_tasks) {
    *data = NULL;
    *dim = 0;
    *extent = NULL;
    return 1;
  }
  if (
      agent->datastate->tasks[task_num]->
      agent_return_data->data_type == -1
     )
  {
    return 1;
  }
  CH_DATATYPE_SIZE(
      agent->datastate->tasks[task_num]->agent_return_data->data_type,
      size);
  num_elements = 1;
  for (
      i = 0; 
      i < agent->datastate->tasks[task_num]->agent_return_data->array_dim;
      i++
      )
  {
    num_elements *= agent->datastate->
      tasks[task_num]->agent_return_data->array_extent[i];
  }


  *data = malloc(num_elements * size);
  memcpy(
      *data,
      agent->datastate->tasks[task_num]->
      agent_return_data->return_data,
      size * num_elements
      );
  *dim = agent->datastate->tasks[task_num]->agent_return_data->array_dim;
  *extent = (int*)malloc(
      sizeof(int) * 
      agent->datastate->tasks[task_num]->agent_return_data->array_dim
      );
  for (i = 0; i < *dim; i++) {
    (*extent)[i] = 
      agent->datastate->tasks[task_num]->agent_return_data->array_extent[i];
  }
  /*    memcpy(
   *extent,
   agent->datastate->tasks[0]->agent_return_data->array_extent,
   *dim
   ); */
  return 0;
} /*}}}*/

int 
MC_GetAgentNumTasks(MCAgent_t agent) /*{{{*/
{
  return agent->datastate->number_of_tasks;
} /*}}}*/

enum MC_AgentType_e
MC_GetAgentType(MCAgent_t agent) /*{{{*/
{
  if (agent != NULL) {
    return agent->agent_type;
  } else {
    return 0;
  }
} /*}}}*/

int 
MC_GetAllAgents(MCAgency_t attr, MCAgent_t **agents, int* num_agents) /*{{{*/
{
  int halt;
  int index = 0;
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  halt = (attr->mc_platform->giant == 1) ? 1 : 0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  if (halt)
    MC_HaltAgency(attr);
  /* Count the number of agents */
  while (agent_queue_SearchIndex(attr->mc_platform->agent_queue, index) != NULL) {
    index++;
  }
  *agents = (MCAgent_t **)malloc(sizeof(MCAgent_t*) * index);
  *num_agents = index;
  /* Assign the agents */
  index = 0;
  while
    (
     (*agents)[index] = agent_queue_SearchIndex
     (
      attr->mc_platform->agent_queue,
      index
     )
    )
    {
      index++;
    }
  if(halt)
    MC_ResumeAgency(attr);
  return 0;
} /*}}}*/



int
MC_HaltAgency(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant=0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

MCAgency_t 
MC_Initialize( /*{{{*/
    int port,
    MCAgencyOptions_t *options)
{
  MCAgency_t ret;
  int i=0;
  int options_malloc = 0;
  ret = (MCAgency_t)malloc(sizeof(struct agency_s));
  if (ret == NULL) {return NULL;}

  ret->hostName = malloc(HOST_NAME_MAX);
  if (ret->hostName == NULL) {return NULL;}
  gethostname(ret->hostName, HOST_NAME_MAX);
  /*Save some memory */
  CHECK_NULL(
      realloc(ret->hostName, sizeof(char)*(strlen(ret->hostName)+1)),
      return NULL;
      );
  if (ret->hostName == NULL) {return NULL;}
  ret->portno = port;
  ret->server = 1;
  ret->client = 0;
  ret->default_agentstatus = -1;
#ifdef MC_SECURITY
  ret->enable_security = 1;
#endif

  /* Set up agency options */
  if(options==NULL) {
    options = (MCAgencyOptions_t*)malloc(sizeof(MCAgencyOptions_t));
    MC_InitializeAgencyOptions(options);
    options_malloc = 1;
  }
  ret->threads = options->threads;
  ret->default_agentstatus = options->default_agent_status;
#ifdef MC_SECURITY
  ret->enable_security = options->enable_security;
#endif
  for(i = 0; i < MC_THREAD_ALL; i++) {
    ret->stack_size[i] = options->stack_size[i];
  }
  /* End agency options */

  ret->mc_platform = mc_platform_Initialize(ret);

  /* Set up the global platform */
  g_mc_platform = ret->mc_platform;

  if (options_malloc)
    free(options);

  return ret;
} /*}}}*/

int 
MC_InitializeAgencyOptions(struct MCAgencyOptions_s* options) /*{{{*/
{
  int i;
  /* Set the default options */
  options->threads = 0xFFFF;
  options->default_agent_status = MC_WAIT_CH;
  options->modified = 0;
#ifdef MC_SECURITY
  options->enable_security = 1;
#endif
  for(i = 0; i < MC_THREAD_ALL; i++) {
    options->stack_size[i] = -1;
  }
  return 0;
} /*}}}*/

int 
MC_MutexLock(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  MUTEX_LOCK(syncnode->lock);
  return 0;
} /*}}}*/

int 
MC_MutexUnlock(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  MUTEX_UNLOCK(syncnode->lock);
  return 0;
} /*}}}*/

int
MC_PrintAgentCode(MCAgent_t agent) /*{{{*/
{
  int progress;
  MUTEX_LOCK(agent->lock);
  progress = agent->datastate->task_progress;
  /* If progress is past the last task, print the last task's code. */
  if (progress >= agent->datastate->number_of_tasks) {
    progress = agent->datastate->number_of_tasks - 1;
  }
  printf("%s\n",
      agent->datastate->agent_code);
  MUTEX_UNLOCK(agent->lock);
  return 0;
} /*}}}*/

int 
MC_RegisterService( /*{{{*/
    MCAgency_t agency,
    MCAgent_t agent,
    int agentID,
    const char *agentName,
    char **serviceNames,
    int numServices)
{
  df_request_list_node_t *req_node;
  df_node_t *new_node;
  int i;
  /* Error checking: Either an agent, or agent Name must be
   * provided. */
  if (agent == NULL && agentName == NULL) {
    return MC_ERR_INVALID_ARGS;
  }
  /* init the request node */
  req_node = df_request_list_node_New();
  req_node->command = (char*)malloc(sizeof(char)*9);
  strcpy((char*)req_node->command, "register");

  new_node = (df_node_t*)malloc(sizeof(df_node_t));
  CHECK_NULL(new_node, return MC_ERR_MEMORY);

  /* Init the lock */
  new_node->lock = (MUTEX_T*)malloc(sizeof(MUTEX_T));
  CHECK_NULL(new_node->lock, return MC_ERR_MEMORY);
  MUTEX_INIT(new_node->lock);

  /* Init the agentID */
  if (agent==NULL) {
    new_node->agent_id = agentID;
  } else {
    new_node->agent_id = agent->id;
  }

  /* Init the agent name */
  if (agent==NULL) {
    new_node->agent_name = 
      (char*)malloc(sizeof(char)*(strlen(agentName)+1));
    CHECK_NULL(new_node->agent_name, return MC_ERR_MEMORY;);
    strcpy(new_node->agent_name, agentName);
  } else {
    new_node->agent_name = 
      (char*)malloc(
          sizeof(char) * 
          (strlen(agent->name)+1)
          );
    CHECK_NULL(new_node->agent_name, return MC_ERR_MEMORY;);
    strcpy(new_node->agent_name, agent->name);
  }

  /* Init the services */
  new_node->service_names = (char**)malloc(
      sizeof(char*) * numServices
      );
  CHECK_NULL(new_node->service_names, return MC_ERR_MEMORY;);
  for (i = 0; i < numServices; i++) {
    new_node->service_names[i] = (char*) malloc(
        sizeof(char) * (strlen(serviceNames[i]) + 1)
        );
    CHECK_NULL(new_node->service_names[i], return MC_ERR_MEMORY;);
    strcpy(
        new_node->service_names[i],
        serviceNames[i]
        );
  }
  new_node->num_services = numServices;

  req_node->data = (void*)new_node;
  req_node->data_size = (sizeof(new_node));

  return df_AddRequest(
      agency->mc_platform->df,
      req_node
      );
} /*}}}*/

int 
MC_ResumeAgency(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 1;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

MCAgent_t
MC_RetrieveAgent(MCAgency_t attr) /*{{{*/
  /* This function retrieves the first agent with agent_status
     MC_AGENT_NEUTRAL it finds. If there are no agents with
     the specified attributes, return value is NULL. */
{
  int i;
  MCAgent_t agent=NULL, ret;
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  for (i = 0; i < attr->mc_platform->agent_queue->size; i++) {
    agent = ListSearch(
        attr->mc_platform->agent_queue->list, i);
    if (agent->agent_status == MC_AGENT_NEUTRAL) {
      break;
    }
  }
  if (agent == NULL) {
    MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
    return NULL;
  }
  if (agent->agent_status != MC_AGENT_NEUTRAL) {
    MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
    return NULL;
  }
  ret = (MCAgent_t)malloc(sizeof(agent_t));
  *ret = *agent;
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return ret;
}/*}}}*/

char * 
MC_RetrieveAgentCode(MCAgent_t agent) /*{{{*/
{
  char *buf;
  int len, progress;
  MUTEX_LOCK(agent->lock);
  progress = agent->datastate->task_progress;
  len = strlen(
      agent->datastate->agent_code);
  buf = (char*)malloc( (len+1)*sizeof(char));
  strcpy(buf,
      agent->datastate->agent_code);
  MUTEX_UNLOCK(agent->lock);
  return buf;
} /*}}}*/

int 
MC_ResetSignal(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 1;
  attr->mc_platform->MC_signal = MC_NO_SIGNAL;
  COND_SIGNAL(attr->mc_platform->giant_cond);
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

int 
MC_SearchForService( /*{{{*/
    /* Input args */
    MCAgency_t attr, 
    const char *searchString,
    /* Return Args */
    char*** agentNames,
    char*** serviceNames,
    int** agentIDs,
    int* numResults)
{
  df_request_search_p search;
  df_search_results_p results;
  df_request_list_node_p request;
  search = df_request_search_New();
  CHECK_NULL(search, return MC_ERR_MEMORY;);
  results = (df_search_results_p)malloc(sizeof(df_search_results_t));
  CHECK_NULL(results, return MC_ERR_MEMORY;);
  request = df_request_list_node_New();
  CHECK_NULL(request, return MC_ERR_MEMORY;);


  search->search_results = results;
  search->search_string = (char*)searchString;

  request->data = (void*)search;
  request->command = malloc(sizeof(char) * 7);
  strcpy((char*)request->command, "search");
  request->data_size = sizeof(df_request_search_t);

  COND_SLEEP_ACTION(
      search->cond,
      search->lock,

      df_AddRequest(attr->mc_platform->df, request);
      );
  /* When we get here, search->results should be filled. */
  *agentNames = search->search_results->agent_names;
  *serviceNames = search->search_results->service_names;
  *agentIDs = search->search_results->agent_ids;
  *numResults = search->search_results->num_results;

  /* Free unused data structures */
  free((void*)request->command);
  df_request_list_node_Destroy(request);
  df_request_search_Destroy(search);

  return MC_SUCCESS;
} /*}}}*/

int 
MC_SemaphorePost(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  SEMAPHORE_POST(syncnode->sem);
  return 0;
} /*}}}*/

int 
MC_SemaphoreWait(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  SEMAPHORE_WAIT(syncnode->sem);
  return 0;
} /*}}}*/

int
MC_SendCh(MCAgency_t attr, /*{{{*/
    const char *filename,
    const char *remotehost,
    int port)
{
  printf("Sorry, not implemented yet.\n");
  return -1;
} /*}}}*/

int 
MC_SendAgentMigrationMessage(MCAgency_t attr, /*{{{*/
    const char *string,
    const char *hostname,
    int port)
{
  message_p message;
  message = message_New();
  if(
      message_InitializeFromString
      (
       attr->mc_platform,
       message,
       string,
       hostname,
       port
      )
    )
  {
    message_Destroy(message);
    return MC_ERR;
  } else {
    return message_queue_Add
      (
       attr->mc_platform->message_queue,
       message
      );
  }
} /*}}}*/

int
MC_SendAgentMigrationMessageFile(MCAgency_t attr,  /*{{{*/
    const char *filename, 
    const char *hostname,
    int port)
{
  struct stat filestat;
  char *buf;
  FILE *fp;
  int ret;
  message_p message;
  extern mc_platform_p g_mc_platform;
  buf = NULL;
  filestat.st_size = 0;
  stat(filename, &filestat);
  if (filestat.st_size != 0 ) {
    buf = malloc( sizeof(char) * (filestat.st_size+1) );
    memset(buf, 0, filestat.st_size+1);
  } else {
    fprintf(stderr, "Error: File %s not found.\n", filename);
    return 1;
  }

  fp = fopen(filename, "r");
  fread((void*)buf, filestat.st_size, 1, fp);
  fclose(fp);

  if (attr!=NULL) {
    message = message_New();
    if( 
        message_InitializeFromString
        (
         attr->mc_platform,
         message,
         buf,
         hostname,
         port
        )
      )
    {
      message_Destroy(message);
    } else {
      ret = message_queue_Add
        (
         attr->mc_platform->message_queue,
         message 
        );
    }
  } else {
    message = message_New();
    if(
        message_InitializeFromString
        (
         g_mc_platform,
         message,
         buf,
         hostname,
         port
        )
      )
    {
      message_Destroy(message);
    } else {
      ret = message_queue_Add
        (
         g_mc_platform->message_queue,
         message
        );
    }
  }
  free(buf);
  return ret;
} /*}}}*/

int 
MC_SendSteerCommand(MCAgency_t attr, enum MC_SteerCommand_e cmd) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->MC_steer_lock);
  attr->mc_platform->MC_steer_command = cmd;
  COND_BROADCAST(attr->mc_platform->MC_steer_cond);
  MUTEX_UNLOCK(attr->mc_platform->MC_steer_lock);
  return 0;
} /*}}}*/

int
MC_SetAgentStatus(MCAgent_t agent, int status) /*{{{*/
{
  MUTEX_LOCK(agent->lock);
  agent->agent_status = status;
  if (!agent->orphan) {
    MUTEX_LOCK(agent->mc_platform->ams->runflag_lock);
    agent->mc_platform->ams->run = 1;
    COND_SIGNAL(agent->mc_platform->ams->runflag_cond);
    MUTEX_UNLOCK(agent->mc_platform->ams->runflag_lock);
  }
  MUTEX_UNLOCK(agent->lock);
  return 0;
} /*}}}*/

int 
MC_SetDefaultAgentStatus(/*{{{*/
    MCAgency_t agency,
    enum MC_AgentStatus_e status
    ) 
{
  agency->mc_platform->default_agentstatus = status;
  return 0;
} /*}}}*/

int 
MC_SetThreadOn(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index) /*{{{*/
{
  SET_THREAD_ON(options->threads, index);
  return 0;
} /*}}}*/

int
MC_SetThreadOff(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index) /*{{{*/
{
  SET_THREAD_OFF(options->threads, index);
  return 0;
} /*}}}*/

int 
MC_Steer(                        /*{{{*/
    MCAgency_t attr,
    int (*funcptr)(void* data),
    void *arg
    )
{
  MUTEX_LOCK(attr->mc_platform->MC_steer_lock);
  do {
    attr->mc_platform->MC_steer_command = MC_RUN;
    MUTEX_UNLOCK(attr->mc_platform->MC_steer_lock);
    (*funcptr)(arg);
  } while 
  (
   attr->mc_platform->MC_steer_command == MC_RESTART 
  );
  return 0;
} /*}}}*/

enum MC_SteerCommand_e 
MC_SteerControl(void) /*{{{*/
{
  extern mc_platform_p g_mc_platform;
  /* Check to see what current command is */
  MUTEX_LOCK(g_mc_platform->MC_steer_lock);
  while (g_mc_platform->MC_steer_command == MC_SUSPEND) {
    COND_WAIT(
        g_mc_platform->MC_steer_cond,
        g_mc_platform->MC_steer_lock
        );
  }
  MUTEX_UNLOCK(g_mc_platform->MC_steer_lock);
  return g_mc_platform->MC_steer_command;
} /*}}}*/

int 
MC_SyncDelete(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *sync_node;
  /* First, lock the entire list. */
  MUTEX_LOCK(attr->mc_platform->syncList->giant_lock);

  /* Find and lock the node */
  sync_node = syncListFind(id, attr->mc_platform->syncList);
  if (sync_node == NULL) {
    MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);
    return MC_ERR_NOT_FOUND;
  }
  MUTEX_LOCK(sync_node->lock);

  /* Remove it from the list so it may no longer be used */
  if (syncListRemove(id, attr->mc_platform->syncList) == NULL) {
    fprintf(stderr, "Fatal error. %s:%d\n",
        __FILE__,
        __LINE__ );
    exit(0);
  }

  /* Now, unlock and destroy */
  MUTEX_UNLOCK(sync_node->lock);
  MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);

  return syncListNodeDestroy(sync_node);
} /*}}}*/

int
MC_SyncInit(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *node;
  node = syncListNodeNew();
  MUTEX_LOCK(attr->mc_platform->syncList->giant_lock);
  if (id == 0) {
    id = rand();
  }
  while (
      syncListFind(id, attr->mc_platform->syncList) != NULL
      ) 
  {
    id = rand();
  }

  node->id = id;
  syncListAddNode(
      node,
      attr->mc_platform->syncList
      );
  MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);
  return id;
}/*}}}*/

int 
MC_TerminateAgent(MCAgent_t agent) /*{{{*/
{
  int status;
  status = Ch_Abort (agent->agent_interp);
  return status;
} /*}}}*/


#ifdef _WIN32
BOOL
MC_MainLoop(MCAgency_t attr) /*{{{*/
{
  /*    return CloseHandle(attr->global->gaf_ap->thread_id[attr->global->mr_index]); */
  Sleep (INFINITE);
  return 0;
}
#else
int 
MC_MainLoop(MCAgency_t attr) 
{
  return pthread_join(attr->mc_platform->ams->thread, NULL);
} /*}}}*/
#endif

int
MC_WaitAgent(MCAgency_t attr) /*{{{*/
{
  int size;
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  while(1) {
    size = attr->mc_platform->agent_queue->size;
    COND_WAIT(
        attr->mc_platform->agent_queue->cond,
        attr->mc_platform->agent_queue->lock
        );
    if (size < attr->mc_platform->agent_queue->size) {
      MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
      break;
    } 
  }
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return 0;
} /*}}}*/

MCAgent_t
MC_WaitRetrieveAgent(MCAgency_t attr) /*{{{*/
{
  int index;
  MCAgent_t agent;
  MC_WaitSignal(attr, MC_RECV_AGENT);
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  index = attr->mc_platform->agent_queue->size-1;
  agent = ListSearch(
      attr->mc_platform->agent_queue->list, index);
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return agent;
} /*}}}*/

/* MC_WaitSignal */
/* This function blocks until one of the signals in argument
 * 'signals' is signalled. 'signals' must be a binary | combination
 * of enum MC_Signal_e type. */
int
MC_WaitSignal(MCAgency_t attr, int signals) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->MC_signal_lock);
  while(! (signals & attr->mc_platform->MC_signal)) {
    COND_WAIT(
        attr->mc_platform->MC_signal_cond,
        attr->mc_platform->MC_signal_lock
        );
  }
  MUTEX_UNLOCK(attr->mc_platform->MC_signal_lock);
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

/* *********************** *
 * Ch Space chdl functions *
 * *********************** */

/* MC_AddAgent */
EXPORTCH int
MC_AddAgent_chdl(void *varg) /*{{{*/
{
  int retval;
  MCAgent_t agent;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agent = (MCAgent_t) Ch_VaArg(interp, ap, void*);
  retval = MC_AddAgent(temp_attr, agent);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CallAgentFunc */
EXPORTCH int
MC_CallAgentFunc_chdl(void *varg) /*{{{*/
{
  int retval;
  /* Function Args */
  MCAgent_t agent;
  const char* funcName;
  void* returnVal;
  void* args;

  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);

  agent =     Ch_VaArg(interp, ap, MCAgent_t);
  funcName =  Ch_VaArg(interp, ap, const char*);
  returnVal = Ch_VaArg(interp, ap, void*);
  args =      Ch_VaArg(interp, ap, void*);

  retval = MC_CallAgentFunc(
      agent,
      funcName,
      returnVal,
      args);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_Barrier_chdl*/
EXPORTCH int
MC_Barrier_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_Barrier(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_BarrierDelete_chdl*/
EXPORTCH int
MC_BarrierDelete_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_BarrierDelete(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_BarrierInit_chdl*/
EXPORTCH int
MC_BarrierInit_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int num_procs;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  num_procs = Ch_VaArg(interp, ap, int);
  retval = MC_BarrierInit(temp_attr, id, num_procs);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondBroadcast_chdl*/
EXPORTCH int
MC_CondBroadcast_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondBroadcast(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondSignal_chdl*/
EXPORTCH int
MC_CondSignal_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondSignal(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondReset_chdl*/
EXPORTCH int
MC_CondReset_chdl(void *varg) /*{{{*/   
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondReset(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondWait_chdl*/
EXPORTCH int
MC_CondWait_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondWait(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

EXPORTCH int
MC_DeleteAgent_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_DeleteAgent(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_DeregisterService_chdl */
EXPORTCH int
MC_DeregisterService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int agentID;
  char *serviceName;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agentID = Ch_VaArg(interp, ap, int);
  serviceName = (char*)Ch_VaArg(interp, ap, const char*);
  retval = MC_DeregisterService(
      temp_attr,
      agentID,
      serviceName );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

EXPORTCH int
MC_End_chdl(void *varg) /* {{{ */
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_End(temp_attr);

  return retval;
} /* }}} */

/* MC_FindAgentByID_chdl*/
EXPORTCH MCAgent_t
MC_FindAgentByID_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_FindAgentByID(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_FindAgentByName_chdl*/
EXPORTCH MCAgent_t
MC_FindAgentByName_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t retval;
  ChInterp_t interp;
  ChVaList_t ap;
  const char *name;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  name = Ch_VaArg(interp, ap, const char *);
  retval = MC_FindAgentByName(temp_attr, name);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_GetAgentArrivalTime */
#ifndef _WIN32
EXPORTCH time_t
#else
EXPORTCH SYSTEMTIME
#endif
MC_GetAgentArrivalTime_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
#ifndef _WIN32
  time_t arrival_time;
#else
  SYSTEMTIME arrival_time;
#endif

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  arrival_time = MC_GetAgentArrivalTime(agent);
  Ch_VaEnd(interp, ap);
  return arrival_time;
} /* }}} */

/* MC_GetAgentID */
EXPORTCH int
MC_GetAgentID_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  id = MC_GetAgentID(agent);
  Ch_VaEnd(interp, ap);
  return id;
} /*}}}*/

/* MC_GetAgentName */
EXPORTCH char*
MC_GetAgentName_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  char* name;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  name = MC_GetAgentName(agent);
  Ch_VaEnd(interp, ap);
  return name;
} /*}}}*/

/* MC_GetAgentNumTasks_chdl */
EXPORTCH int
MC_GetAgentNumTasks_chdl(void *varg)
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  int num_tasks;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  num_tasks = MC_GetAgentNumTasks(agent);
  Ch_VaEnd(interp, ap);
  return num_tasks;
}

/* MC_GetAgentStatus_chdl */
EXPORTCH int
MC_GetAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  int status;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  status = MC_GetAgentStatus(agent);
  Ch_VaEnd(interp, ap);
  return status;
} /*}}}*/

/* MC_GetAgentXMLString_chdl */
EXPORTCH char *
MC_GetAgentXMLString_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  char *retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_GetAgentXMLString(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

#ifndef _WIN32
EXPORTCH int
MC_GetTimeOfDay_chdl(void *varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct timeval *tv;
  Ch_VaStart(interp, ap, varg);
  tv = Ch_VaArg(interp, ap, struct timeval*);
  gettimeofday(tv, NULL);
  Ch_VaEnd(interp, ap);
  return 0;
}
#endif

/* MC_HaltAgency_chdl */
EXPORTCH int
MC_HaltAgency_chdl(void *varg)
{
  MCAgency_t temp_attr;
  int retval;
  extern mc_platform_p g_mc_platform;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_HaltAgency(temp_attr);

  free(temp_attr);
  return retval;
}

/* MC_MutexLock_chdl */
EXPORTCH int
MC_MutexLock_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_MutexLock(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_MutexUnlock_chdl */
EXPORTCH int
MC_MutexUnlock_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_MutexUnlock(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_PrintAgentCode_chdl */
EXPORTCH int
MC_PrintAgentCode_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_PrintAgentCode(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_RegisterService_chdl */
EXPORTCH int
MC_RegisterService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  int retval;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  /* varg arguments */
  MCAgent_t agent;
  char **serviceNames;
  int numServices;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  serviceNames = Ch_VaArg(interp, ap, char **);
  numServices = Ch_VaArg(interp, ap, int);

  retval = MC_RegisterService(
      temp_attr,      /* agency */
      agent,          /* agent */ 
      0,              /* agent id */
      NULL,           /* agent name */
      serviceNames,   /* services */
      numServices
      );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_ResumeAgency_chdl */
EXPORTCH int
MC_ResumeAgency_chdl(void *varg)
{
  MCAgency_t temp_attr;
  int retval;
  extern mc_platform_p g_mc_platform;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_ResumeAgency(temp_attr);

  free(temp_attr);
  return retval;
}

/* MC_RetrieveAgent_chdl */
EXPORTCH MCAgent_t
MC_RetrieveAgent_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t agent;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  agent = MC_RetrieveAgent(temp_attr);
  free(temp_attr);
  return agent;
} /*}}}*/

/* MC_RetrieveAgentCode_chdl */
EXPORTCH char *
MC_RetrieveAgentCode_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  char *retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_RetrieveAgentCode(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_SearchForService_chdl */
EXPORTCH int
MC_SearchForService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  int retval;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  /* Args */
  const char* searchString;
  char*** agentNames;
  char*** serviceNames;
  int** agentIDs;
  int* numResults;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  searchString = Ch_VaArg(interp, ap, const char*);
  agentNames = Ch_VaArg(interp, ap, char***);
  serviceNames = Ch_VaArg(interp, ap, char***);
  agentIDs = Ch_VaArg(interp, ap, int**);
  numResults = Ch_VaArg(interp, ap, int*);

  retval = MC_SearchForService(
      temp_attr,
      searchString,
      agentNames,
      serviceNames,
      agentIDs,
      numResults
      );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SemaphorePost_chdl */
EXPORTCH int
MC_SemaphorePost_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_SemaphorePost(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SemaphoreWait_chdl */
EXPORTCH int 
MC_SemaphoreWait_chdl(void *varg) /*{{{*/
{   
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_SemaphoreWait(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SendAgentMigrationMessage_chdl*/
EXPORTCH int
MC_SendAgentMigrationMessage_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  const char *message, *hostname;
  int port, retval;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = NULL;

  Ch_VaStart(interp, ap, varg);
  message = Ch_VaArg(interp, ap, char *);
  hostname = Ch_VaArg(interp, ap, char *);
  port = Ch_VaArg(interp, ap, int);
  retval = MC_SendAgentMigrationMessage(temp_attr, message, hostname, port);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_SendAgentMigrationMessageFile_chdl*/
EXPORTCH int
MC_SendAgentMigrationMessageFile_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  char *filename, *hostname;
  int port, retval;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = NULL;

  Ch_VaStart(interp, ap, varg);
  filename = Ch_VaArg(interp, ap, char *);
  hostname = Ch_VaArg(interp, ap, char *);
  port = Ch_VaArg(interp, ap, int);
  retval = MC_SendAgentMigrationMessageFile(temp_attr, filename, hostname, port);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

EXPORTCH int
MC_SendSteerCommand_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  int retval;
  enum MC_SteerCommand_e command;
  extern mc_platform_p g_mc_platform;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  command = Ch_VaArg(interp, ap, enum MC_SteerCommand_e );
  retval = MC_SendSteerCommand(temp_attr, command);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SetAgentStatus_chdl */
EXPORTCH int
MC_SetAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  int status;
  int ret;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  status = Ch_VaArg(interp, ap, int);
  ret = MC_SetAgentStatus(agent, status);
  Ch_VaEnd(interp, ap);
  return ret;
} /*}}}*/

/* MC_SetDefaultAgentStatus_chdl */
EXPORTCH int
MC_SetDefaultAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int status;
  int ret;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(1););
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  status = Ch_VaArg(interp, ap, int);
  ret = MC_SetDefaultAgentStatus(temp_attr, status);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return ret;
} /*}}}*/

/* MC_SyncDelete_chdl*/
EXPORTCH int
MC_SyncDelete_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_SyncDelete(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SyncInit_chdl*/
EXPORTCH int
MC_SyncInit_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);

  retval = MC_SyncInit(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_TerminateAgent_chdl */
EXPORTCH int
MC_TerminateAgent_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_TerminateAgent(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

