/*[
 * Copyright (c) 2007 Integration Engineering Laboratory
                      University of California, Davis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the complete modified source code.  Modifications are to
 * be distributed as patches to the released version.  Permission to
 * distribute binaries produced by compiling modified sources is granted,
 * provided you
 *   1. distribute the corresponding source modifications from the
 *    released version in the form of a patch file along with the binaries,
 *   2. add special version identification to distinguish your version
 *    in addition to the base release version number,
 *   3. provide your name and address as the primary contact for the
 *    support of your modified version, and
 *   4. retain our contact information in regard to use of the base
 *    software.
 * Permission to distribute the released version of the source code along
 * with corresponding source modifications in the form of a patch file is
 * granted with same provisions 2 through 4 for binary distributions.
 *
 * This software is provided "as is" without express or implied warranty
 * to the extent permitted by applicable law.
]*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "include/fipa_acl_message.h"
#include "include/mc_error.h"
#include "include/macros.h"

int fipa_acl_Parse(fipa_acl_message_p acl, fipa_message_string_p message)
{
  int err = 0;
  if (fipa_GetAtom(message,'(')) {
    err = MC_ERR_PARSE;
    goto exit;
  }
  if (fipa_message_type_Parse(&(acl->performative), message)) {
    err = MC_ERR_PARSE;
    goto exit;
  }
  while(!fipa_message_parameter_Parse(acl, message));

exit:
  return err;
}

int fipa_message_parameter_Parse(fipa_acl_message_p acl, fipa_message_string_p message)
{
  int err;
  fipa_word_t* word;
  char* parameter;
  if((err = fipa_GetAtom(message, ':'))) return err;
  if((err = fipa_word_Parse(&word, message))) return err;
  parameter = word->content;
  if (!strcmp(parameter, "sender")) {
    err = fipa_agent_identifier_Parse(&(acl->sender), message);
  } else if (!strcmp(parameter, "receiver")) {
    err = fipa_agent_identifier_set_Parse(&(acl->receiver), message);
  } else if (!strcmp(parameter, "content")) {
    err = fipa_string_Parse(&(acl->content), message);
  } else if (!strcmp(parameter, "reply-with")) {
    err = fipa_expression_Parse(&(acl->reply_with), message);
  } else if (!strcmp(parameter, "reply-by")) {
    err = fipa_datetime_Parse(&(acl->reply_by), message);
  } else if (!strcmp(parameter, "in-reply-to")) {
    err = fipa_expression_Parse(&(acl->in_reply_to), message);
  } else if (!strcmp(parameter, "reply-to")) {
    err = fipa_agent_identifier_set_Parse(&(acl->reply_to), message);
  } else if (!strcmp(parameter, "language")) {
    err = fipa_expression_Parse(&(acl->language), message);
  } else if (!strcmp(parameter, "encoding")) {
    err = fipa_expression_Parse(&(acl->encoding), message);
  } else if (!strcmp(parameter, "ontology")) {
    err = fipa_expression_Parse(&(acl->ontology), message);
  } else if (!strcmp(parameter, "protocol")) {
    err = fipa_word_Parse(&(acl->protocol), message);
  } else if (!strcmp(parameter, "conversation-id")) {
    err = fipa_expression_Parse(&(acl->conversation_id), message);
  } else {
    /* FIXME: We do not deal with user defined parameters yet. */
    fprintf(stderr, "FIXME: No handling of user defined parameters. %s:%d\n",
        __FILE__, __LINE__);
    err = MC_ERR_PARSE;
  }
  return err;
}

int fipa_message_type_Parse(
    enum fipa_performative_e* performative, 
    fipa_message_string_p message
    )
{
  int err = 0;
  fipa_word_p word;
  err = fipa_word_Parse(&word, message);
  if (err) return err;
  if(!strcmp(word->content, "accept-proposal")) {
    *performative = FIPA_ACCEPT_PROPOSAL;
  } else if (!strcmp(word->content, "agree")) {
    *performative = FIPA_AGREE;
  } else if (!strcmp(word->content, "cancel")) {
    *performative = FIPA_CANCEL;
  } else if (!strcmp(word->content, "call-for-proposal")) {
    *performative = FIPA_CALL_FOR_PROPOSAL;
  } else if (!strcmp(word->content, "confirm")) {
    *performative = FIPA_CONFIRM;
  } else if (!strcmp(word->content, "disconfirm")) {
    *performative = FIPA_DISCONFIRM;
  } else if (!strcmp(word->content, "failure")) {
    *performative = FIPA_FAILURE;
  } else if (!strcmp(word->content, "inform")) {
    *performative = FIPA_INFORM;
  } else if (!strcmp(word->content, "inform-if")) {
    *performative = FIPA_INFORM_IF;
  } else if (!strcmp(word->content, "inform-ref")) {
    *performative = FIPA_INFORM_REF;
  } else if (!strcmp(word->content, "not-understood")) {
    *performative = FIPA_NOT_UNDERSTOOD;
  } else if (!strcmp(word->content, "propogate")) {
    *performative = FIPA_PROPOGATE;
  } else if (!strcmp(word->content, "propose")) {
    *performative = FIPA_PROPOSE;
  } else if (!strcmp(word->content, "proxy")) {
    *performative = FIPA_PROXY;
  } else if (!strcmp(word->content, "query-if")) {
    *performative = FIPA_QUERY_IF;
  } else if (!strcmp(word->content, "query-ref")) {
    *performative = FIPA_QUERY_REF;
  } else if (!strcmp(word->content, "refuse")) {
    *performative = FIPA_REFUSE;
  } else if (!strcmp(word->content, "reject-proposal")) {
    *performative = FIPA_REJECT_PROPOSAL;
  } else if (!strcmp(word->content, "request")) {
    *performative = FIPA_REQUEST;
  } else if (!strcmp(word->content, "request-when")) {
    *performative = FIPA_REQUEST_WHEN;
  } else if (!strcmp(word->content, "request-whenever")) {
    *performative = FIPA_REQUEST_WHENEVER;
  } else if (!strcmp(word->content, "subscribe")) {
    *performative = FIPA_SUBSCRIBE;
  } else {
    fprintf(stderr, "Unknown performative: '%s'. %s:%d\n",
        word->content, __FILE__, __LINE__);
    err = MC_ERR_PARSE;
  }
  return err;
}

int fipa_GetAtom(
    fipa_message_string_p message,
    char expected_atom
    )
{
  while
    (
     (*(message->parse) >= 0x00) &&
     (*(message->parse) <= 0x20)
    )
    {
      if (*(message->parse) == 0x00)
        return MC_ERR_PARSE;
      message->parse++;
    }
  if( *(message->parse) == expected_atom) {
    message->parse++;
    return MC_SUCCESS;
  } else {
    return MC_ERR_PARSE;
  }
}

int fipa_word_Parse(fipa_word_t** word, fipa_message_string_p message)
{
  char* tmp;
  int i = 0;
  /* Get rid of leading whitespace */
  while
    (
     (*(message->parse)>=0x00) &&
     (*(message->parse)<=0x20)
    )
    {
      /* If we encounter a null zero, return error. */
      if (*(message->parse) == '\0') {
        return MC_ERR_PARSE;
      }
      message->parse++;
    }
  /* Count number of characters in word */
  tmp = message->parse;
  while (*tmp > 0x20) {
    tmp++;
    i++;
  }
  *word = (fipa_word_t*)malloc(sizeof(fipa_word_t));
  CHECK_NULL(*word, exit(0););
  (*word)->content = malloc
    (
     sizeof(char) * (i+1)
    );
  CHECK_NULL((*word)->content, exit(0););

  /* Copy word */
  i = 0;
  while( *(message->parse) > 0x20 ) {
    ((*word)->content)[i] = *(message->parse);
    message->parse++;
    i++;
  }
  ((*word)->content)[i] = '\0';
  return MC_SUCCESS;
}

int fipa_word_Destroy(fipa_word_t* word)
{
  if(!word)
    return MC_SUCCESS;
  if (word->content)
    free(word->content);
  free(word);
  return MC_SUCCESS;
}

int fipa_CheckNextToken(const fipa_message_string_p message, const char* token)
{
  char* tmp = message->parse;
  while 
    (
     (*tmp >= 0x00) &&
     (*tmp <= 0x20)
    )
      tmp++;
  while (*token != '\0') {
    if (*token != *tmp) {
      return 0; /* False value, not error code */
    }
    token++;
    tmp++;
  }
  return 1; /* True */
}

int fipa_expression_Parse(fipa_expression_t** expression, fipa_message_string_p message)
{
  int i=0;
  *expression = (fipa_expression_t*)malloc(sizeof(fipa_expression_t));
  /* The expression may contain a word, string, date, or list of expressions. First,
   * lets check the recursive case, which is a parentheses-bound list of 
   * expressions. */
  if (fipa_CheckNextToken(message, "(")) {
    (*expression)->type = FIPA_EXPR_EXPRESSION;
    if(fipa_GetAtom(message, '(')) {
      /* This should never happen */
      fprintf(stderr, "Fatal error. %s:%d\n", __FILE__, __LINE__);
      exit(0);
    }
    for 
      (
       i = 0; 
       !fipa_expression_Parse( &(((*expression)->content.expression)[i]), message);
       i++
      );
    if(fipa_GetAtom(message, ')')) {
      fprintf(stderr, "Parse error. %s:%d\n", __FILE__, __LINE__);
      return MC_ERR_PARSE;
    }
  } else if (
      /* The expression may be a date-time */
      !fipa_datetime_Parse(&((*expression)->content.datetime), message)
      ) 
  {
    (*expression)->type = FIPA_EXPR_DATETIME;
  } else if (
      /* The expression may be a string */
      !fipa_string_Parse(&((*expression)->content.string), message)
      )
  {
    (*expression)->type = FIPA_EXPR_STRING;
  } else if (
      /* The expression may be a word */
      !fipa_word_Parse(&((*expression)->content.word), message)
      )
  {
    (*expression)->type=FIPA_EXPR_WORD;
  }
  else
  {
    /* It's not correct */
    return MC_ERR_PARSE;
  }
  return MC_SUCCESS;
}

int fipa_GetNextWord(char** word, const fipa_message_string_p message)
{
  char *tmp;
  int i = 0;
  int j;
  /* Skip leading whitespace */
  tmp = message->parse;
  while
    (
     (*tmp >= 0x00) &&
     (*tmp <= 0x20)
    )
      tmp++;
  /* See how big the word is */
  /* The first character has special rules */
  if
    (
     ((*tmp >= 0x00) && (*tmp <= 0x20)) ||
     (*tmp == '(') ||
     (*tmp == ')') ||
     (*tmp == '#') ||
     ((*tmp >= 0x30) && (*tmp <= 0x39)) || /* May not start with a digit */
     (*tmp == '-') ||
     (*tmp == '@')
    )
      return ERR;
  i++;
  tmp++;
  /* Count the rest of the chars */
  while
    (
     ((*tmp < 0x00) || (*tmp > 0x20)) &&
     (*tmp != '(') &&
     (*tmp != ')')
    ) {
      i++;
      tmp++;
    }
  /* Allocate the memory */
  *word = (char*)malloc(sizeof(char) * (i+1));

  for (j = 0; j < i; j++) {
    *((*word) + j) = *(message->parse+j);
  }
  *((*word)+j) = '\0';
  return MC_SUCCESS;
}

int fipa_datetime_Parse(fipa_DateTime_p* datetime, fipa_message_string_p message)
{
  char *word;
  char *tmp;
  int i;
  char buf[5];
  /* First, lets get the word, and see if it is indeed a datetime. */
  fipa_GetNextWord(&word, message);
  tmp = word;
  if (
      (*tmp == '+') ||
      (*tmp == '-')
     )
    tmp++;
  /* The next 8 characters must be digits */
  for(i = 0; i < 8; i++) {
    if (*tmp < 0x30 || *tmp > 0x39) {
      free(word);
      return MC_ERR_PARSE;
    }
    tmp++;
  }
  /* Next character must be 'T' */
  if (*tmp == 'T') {
    tmp++;
  } else {
    free(word);
    return MC_ERR_PARSE;
  }
  /* Next 9 characters must be digits */
  for(i = 0; i < 9; i++) {
    if(*tmp < 0x30 || *tmp > 0x39) {
      free(word);
      return MC_ERR_PARSE;
    }
    tmp++;
  }

  /* If we get here, the string is definately a date-time. */
  *datetime = (fipa_DateTime_p)malloc(sizeof(fipa_DateTime_t));
  tmp = word;
  switch(*tmp) {
    case '+':
      (*datetime)->sign = '+';
      tmp++;
      break;
    case '-':
      (*datetime)->sign = '-';
      tmp++;
      break;
    default:
      break;
  }

  /* Get the year */
  for(i = 0; i < 4; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->year = atoi(buf);

  /* Get the month */
  for(i = 0; i < 2; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->month = atoi(buf);

  /* Get the day */
  for(i = 0; i < 2; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->month = atoi(buf);

  /* Skip the T */
  if (*tmp != 'T') {
    /* Something is very wrong */
    fprintf(stderr, "Fatal Error. %s:%d\n", __FILE__, __LINE__);
    exit(0);
  }
  tmp++;

  /* Get the hour */
  for(i = 0; i < 2; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->hour = atoi(buf);

  /* Get the minute */
  for(i = 0; i < 2; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->minute = atoi(buf);

  /* Get the second */
  for(i = 0; i < 2; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->second = atoi(buf);

  /* Get the millisecond */
  for(i = 0; i < 3; i++) {
    buf[i] = *tmp;
    tmp++;
  }
  buf[i] = '\0';
  (*datetime)->millisecond = atoi(buf);

  if (*tmp == 'Z')
    (*datetime)->is_utc = 1;
  else
    (*datetime)->is_utc = 0;
  return MC_SUCCESS;
}

int fipa_string_Parse( fipa_string_p* fipa_string, fipa_message_string_p message)
{
  int len;
  int word_count;
  char *tmp;
  /* Get rid of leading whitespace */
  while
    (
     (*(message->parse) >= 0x00) &&
     (*(message->parse) <= 0x20)
    )
      message->parse++;
  /* Read each character until newline or end of string. */
  /* FIXME: We also need to treat '\' as an escape character. */
  /* First, count number of characters so we know how much to allocate. */
  tmp = message->parse;
  len = 0;
  while 
    (
     (*tmp != '\n') &&
     (*tmp != '\0')
    )
    {
      if
        (
         (*tmp >= 0x00) &&
         (*tmp <= 0x20)
        )
          word_count++;
      tmp++;
      len++;
    }
  *fipa_string = (fipa_string_p)malloc(sizeof(fipa_string_t));
  (*fipa_string)->content = (char*)malloc
    (
     sizeof(char) * (len + 1)
    );
  len = 0;
  while (message->parse < tmp) {
    ((*fipa_string)->content)[len] = *(message->parse);
    len++;
    message->parse++;
  }
  ((*fipa_string)->content)[len] = '\0';
  return MC_SUCCESS;
}

int fipa_agent_identifier_Parse(fipa_agent_identifier_p* aid, fipa_message_string_p message)
{
  int err;
  fipa_word_t* word;
  char *rewind;
  if
    (
     (err = fipa_GetAtom(message, '(') ) 
    ) return err;
  if
    (
     (err = fipa_word_Parse(&word, message) )
    ) return err;
  if (strcmp(word->content, "agent-identifier")) {
    free(word->content);
    free(word);
    return MC_ERR_PARSE;
  }
  fipa_word_Destroy(word);
  if 
    (
     (err = fipa_GetAtom(message, ':') )
    ) return err;
  if
    (
     (err = fipa_word_Parse(&word, message))
    ) return err;
  if (strcmp(word->content, "name")) {
    return MC_ERR_PARSE;
  }
  fipa_word_Destroy(word);
  /* This is probably a valid aid, so allocate it. */
  *aid = (fipa_agent_identifier_p)malloc(sizeof(fipa_agent_identifier_t));
  if
    (
     (err = fipa_word_Parse(&word, message))
    ) return err;
  (*aid)->name = (char*)malloc
    (
     sizeof(char) * 
     (strlen(word->content)+1)
    );
  CHECK_NULL((*aid)->name, exit(0););
  strcpy((*aid)->name, word->content);

  /* Now we need to see if there are addresses and/or resolvers. */

  rewind = message->parse;
  if
    (
     (err = fipa_GetAtom(message, ':') )
    ) return MC_ERR_PARSE;
  if
    (
     (err = fipa_word_Parse(&word, message))
    ) {
      message->parse = rewind;
      return MC_ERR_PARSE;
    }

  if (!strcmp(word->content, "addresses"))
  {
    err = fipa_url_sequence_Parse(&((*aid)->addresses), message);
    if(err) {
      message->parse = rewind;
      return err;
    }
  } else {
    message->parse = rewind;
  }
  return MC_SUCCESS;
  /* FIXME: We will deal with resolvers and custom fields later */
}

int fipa_agent_identifier_set_Parse(fipa_agent_identifier_set_p* agent_ids, fipa_message_string_p message)
{
  int err;
  fipa_word_p word;
  int i=0;
  /* FIXME */
  if
    (
     (err = fipa_GetAtom(message, '(') )
    ) return err;
  if
    (
     (err = fipa_word_Parse(&word, message) )
    ) return err;
  if (strcmp(word->content, "set")) {
    free(word->content);
    free(word);
    return MC_ERR_PARSE;
  }
  free(word->content);
  free(word);
  *agent_ids = (fipa_agent_identifier_set_p)malloc(sizeof(struct fipa_agent_identifier_set_s));
  (*agent_ids)->fipa_agent_identifiers = 
    (fipa_agent_identifier_p*)malloc(20 * sizeof(fipa_agent_identifier_t*));
  while 
    ( 
     (
      err = fipa_agent_identifier_Parse
      (&(((*agent_ids)->fipa_agent_identifiers)[i]), message)
     ) == MC_SUCCESS
    )
    {
      i++;
    }
}

int fipa_url_sequence_Parse(fipa_url_sequence_p* urls, fipa_message_string_p message)
{

  return 0;
}
