using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

/** \file /LibMCInterop/Cs2Ch.cs
 * Class file for importing functions from cs2ch.dl.
 */

namespace LibMCInterop
{
    /**
     * \brief     Wrapper class for the example agent interface.
     * 
     * Contains structure definitions, imports from DL file, and
     * wrapper functions for imported functions.
     */
    public class Cs2Ch
    {
        /**
         * \brief     Path to the dl file built by the cs2ch example
         * 
         * If you have not installed Ch to the default directory, 
         * your main hard drive is not "C:", or you have installed
         * cs2ch in a directory other than the default, you will need 
         * to change this to make the project work correctly.
         */
        private const String dlfile =
            "C:\\Ch\\toolkit\\embedch\\package\\cs2ch\\dl\\libcs2ch.dl";    // Default install location
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\Debug\\libcs2ch.dl"; // Debug build from VS
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\Release\\libcs2ch.dl"; // Release build from VS
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\cs2ch\\dl\\libcs2ch.dl"; // dlcomp/dllink build from package

        /**
         * \brief     Delegate function for callback in cs2ch.dl
         * 
         * This is the delegate function for the callback used by 
         * setCallback() and callFptr() in cs2ch.dl. It is the rough 
         * equivalent of "typedef int (__stdcall *Fx)(char*)". See the
         * documentation for interoperability for more information.
         */
        public delegate int Fx(String s);

        /**
         * \brief     Delegate function for callback in cs2ch.dl
         * 
         * This is the delegate function for the callback used by 
         * setFunc2Callback() and callFunc2() in cs2ch.dl. It is the rough 
         * equivalent of "typedef int (__stdcall *Fx)(arg_struct*)". See the
         * documentation for interoperability for more information.
         */
        public delegate int Fy(ref arg_struct s);

        /**
         * \brief     Argument structure for agent function "hello."
         * 
         * Defined in the persistent.xml agent example.
         */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        public struct arg_struct
        {
            public int a;
            public int b;
        }

        /**
         * \brief     A structure from cs2ch.dl
         * 
         * Includes an int, double, fixed length array, and 
         * a String.
         */
        [StructLayout(LayoutKind.Sequential)]
        public struct struct1_t
        {
            public int i;
            public double d;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public int[] iarr;
            public int uses;
            public String name;

            public override string ToString()
            {
                return "i = " + i.ToString() + "\n" +
                       "d = " + d.ToString() + "\n" +
                       "uses = " + uses.ToString() + "\n" +
                       "name = " + name;
            }
        };

        /**
         * \brief     A structure from cs2ch.dl
         * 
         * Includes an int, double, fixed length array, and 
         * a String, and a pointer to another structure.
         */
        [StructLayout(LayoutKind.Sequential)]
        public struct struct2_t
        {
            public int i;
            public double d;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public int[] iarr;
            public IntPtr s; // pointer to struct struct1_t
            public int uses;
            public String name;

            public override string ToString()
            {
                return "i = " + i.ToString() + "\n" +
                       "d = " + d.ToString() + "\n" +
                       "uses = " + uses.ToString() + "\n" +
                       "name = " + name;
            }
        };

        /**
         * \brief     Sets the callback in the library.
         * 
         * Sets the callback in the cs2ch library. Handles 
         * creating the function pointer from the delegate
         * automatically.
         * 
         * \param     callback Fx delegate to use as callback
         */
        public static void SetFxCallBack(Fx callback)
        {
            //IntPtr f = Marshal.GetFunctionPointerForDelegate(callback);
            Cs2Ch.setFxCallback(callback);//f);
        }

        /**
         * \brief     Sets the callback in the library.
         * 
         * Sets the callback in the cs2ch library. Handles 
         * creating the function pointer from the delegate
         * automatically.
         * 
         * \param     callback Fy delegate to use as callback
         */
        public static void SetFyCallBack(Fy callback)
        {
            //IntPtr f = Marshal.GetFunctionPointerForDelegate(callback);
            Cs2Ch.setFyCallback(callback);//f);
        }

        public static struct1_t NewStruct1()
        {
            IntPtr sp = new_struct1();
            struct1_t s = (struct1_t)Marshal.PtrToStructure(sp, typeof(struct1_t));
            free_struct1(sp);
            return s;
        }

        public static struct2_t Func3(ref struct1_t s1, out struct1_t embedded_s1)
        {
            embedded_s1 = new struct1_t();
            IntPtr sp = func3(ref s1);
            if (sp != IntPtr.Zero)
            {
                struct2_t s = (struct2_t)Marshal.PtrToStructure(sp, typeof(struct2_t));
                /*
                 * This needs help: sp returned by func 3 has a pointer to a struct1_t
                 * that was set to the "ref s1" passed into func3. "ref s1" is not a valid
                 * pointer, so how do we handle that? Temporarily, the embedded IntPtr in
                 * s is set to zero to avoid freeing problems.
                 * */
                embedded_s1 = (struct1_t)Marshal.PtrToStructure(s.s, typeof(struct1_t));
                free_struct2(sp);
                return s;
            }
            else
                return new struct2_t();
        }

        public static struct2_t Func4(struct1_t s1)
        {
            IntPtr sp = func4(s1);
            if (sp != IntPtr.Zero)
            {
                struct2_t s = (struct2_t)Marshal.PtrToStructure(sp, typeof(struct2_t));
                free_struct2(sp);
                return s;
            }
            else
                return new struct2_t();
        }

        [DllImport(dlfile)]
        public static extern void dll_init();
        [DllImport(dlfile)]
        public static extern void dll_exit();
        [DllImport(dlfile)]
        public static extern int get_inc_int();
        [DllImport(dlfile)]
        public static extern int get_int();
        [DllImport(dlfile)]
        public static extern int add(int a, int b);
        [DllImport(dlfile)]
        public static extern double doublesum(double a, double b);
        [DllImport(dlfile)]
        public static extern int func1([MarshalAs(UnmanagedType.Struct)] struct1_t s1, [MarshalAs(UnmanagedType.Struct)] struct2_t s2);
        [DllImport(dlfile)]
        public static extern double func2(ref struct1_t s1, ref struct2_t s2);
        [DllImport(dlfile)]
        public static extern void callFxCallback(String s);
        [DllImport(dlfile)]
        public static extern void callFyCallback(ref arg_struct arg);

        // These functions can be called directly, but wrappers are provided as well.
        [DllImport(dlfile)]
        public static extern void setFxCallback(Fx f);
        [DllImport(dlfile)]
        public static extern void setFyCallback(Fy f);

        // These functions require manual marshalling or involve unmanaged dynamic memory, so they will be wrapped
        [DllImport(dlfile)]
        private static extern IntPtr func3(ref struct1_t s1);
        [DllImport(dlfile)]
        private static extern IntPtr func4(struct1_t s1);
        [DllImport(dlfile)]
        private static extern IntPtr new_struct1();
        [DllImport(dlfile)]
        private static extern void free_struct1(IntPtr s);
        [DllImport(dlfile)]
        private static extern void free_struct2(IntPtr s);

    }
}
