/**
* \file cs2chfunc.c
* This file contains function definitions for the cs2ch
* library
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "cs2ch.h"    /* header file for cs2ch lib */

/* Macro for safe string copy under Windows */
#ifdef _CH_
#define strcpy_s(dest, len, src) strcpy(dest, src)
#endif

// A function pointer for the callback.
Fx fptr = NULL;
Fy func2ptr = NULL;

// Static variables to show persistence of the library in memory.
int count = 0;
struct1_t *dll_s1 = NULL;
struct2_t *dll_s2 = NULL;
char* n1 = "Name 1";
char* n2 = "Name 2";

/* This is an optional function for Windows only that 
   allows you to specify DLL entry and exit routines */
/*
#ifdef _WIN32
#include <windows.h>
BOOL WINAPI DllMain(
  HANDLE hinstDLL, 
  DWORD dwReason, 
  LPVOID lpvReserved
)
{
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);
	printf("Dll main running.\n");
	fflush(stdout);
    count = 0;
	return TRUE; // Return FALSE to abort loading library
}
#endif
*/

/**
 * \brief     Sets a callback pointer in the library.
 * 
 * Calls to this library can use the callback pointer
 * to call any function with the appropriate signature.
 *
 * \param	  f An Fx callback pointer.
 * \returns   1 if the callback is assigned, 0 otherwise.
 */
EXPORTCH int setCallback(Fx f)
{
	if (f != NULL)
	{
		fptr = f;
        return TRUE;
	}
    else
    {
        return FALSE;
    }
}

/**
 * \brief     Calls the callback pointer in the library.
 * 
 * Function used to call the callback pointer stored in 
 * the library.
 *
 * \param	  s A string to pass to the callback
 * \returns   The return value of the callback function
 */
EXPORTCH int callFptr(char* s)
{
	int i = 0;
	if (fptr != NULL)
	{
        if (s != NULL)
        {
            i = (*fptr)(s);
            printf("Returned %i\n", i);
            fflush(stdout);
        }
	}
    else
    {
        fprintf(stderr, "Callback function pointer not set!\n");
        fflush(stderr);
    }
    return i;
}

/**
 * \brief     Sets a callback pointer in the library.
 * 
 * Calls to this library can use the callback pointer
 * to call any function with the appropriate signature.
 *
 * \param	  f An Fy callback pointer.
 * \returns   1 if the callback is assigned, 0 otherwise.
 */
EXPORTCH int setFunc2Callback(Fy f)
{
	if (f != NULL)
	{
		func2ptr = f;
        return TRUE;
	}
    else
    {
        return FALSE;
    }
}

/**
 * \brief     Calls the callback pointer in the library.
 * 
 * Function used to call the callback pointer stored in 
 * the library.
 *
 * \param	  s A pointer to an arg_struct to pass to the callback
 * \returns   The return value of the callback function
 */
EXPORTCH int callFunc2(arg_struct* s)
{
	int i = 0;
	if (func2ptr != NULL)
	{
        if (s != NULL)
        {
            i = (*func2ptr)(s);
            printf("Returned %i\n", i);
            fflush(stdout);
        }
	}
    else
    {
        fprintf(stderr, "Callback function pointer not set!\n");
        fflush(stderr);
    }
    return i;
}

/**
 * \brief     Returns and increments a static int.
 * 
 * This function returns the value of an int stored
 * in the library and increments it. Can be used to 
 * show persistence in the library.
 *
 * \returns   The (unincremented) value of the static int
 */
EXPORTCH int get_inc_int(void)
{
    printf("Count = %i.\n", count);
    fflush(stdout);
    return count++;
}

/**
 * \brief     Returns the value of a static int.
 * 
 * This function returns the value of an int stored
 * in the library.
 *
 * \returns   The value of the static int
 */
EXPORTCH int get_int(void)
{
    printf("Count = %i.\n", count);
    fflush(stdout);
    return count;
}

EXPORTCH int dll_init(void)
{
	count = 0;

	dll_s1 = (struct1_t*) malloc(sizeof(struct1_t));
	dll_s1->d = 0;
	dll_s1->i = 0;
	dll_s1->uses = -1;
	dll_s1->name = n1;
	memset(dll_s1->iarr, -1, sizeof(dll_s1->iarr));

	dll_s2 = (struct2_t*) malloc(sizeof(struct2_t));
	dll_s2->d = 3.14159;
	dll_s2->i = 314159;
	dll_s2->uses = 0;
	dll_s2->s = dll_s1;
	dll_s2->name = n2;
	memset(dll_s2->iarr, -1, sizeof(dll_s2->iarr));
    
    return count;
}

EXPORTCH int dll_exit(void)
{
	if (dll_s1 != NULL)
		free(dll_s1);
	if (dll_s2 != NULL)
		free(dll_s2);
    return count;
}

/**
 * \brief     Simple function that adds to ints.
 * 
 * This function adds two ints and returns the
 * result.
 *
 * \returns   The sum of the arguments.
 */
EXPORTCH int add(int a, int b)
{
	int retval;
	retval = a + b;
	printf("In sum: a = %i, b = %i\n", a, b);
	printf("Returning %i\n", retval);
	fflush(stdout);

	fprintf(stderr, "This is an error.\n");
	fflush(stderr);

	return retval;
}

/**
 * \brief     Simple function that adds to doubles.
 * 
 * This function adds two doubles returns the
 * result.
 *
 * \returns   The sum of the arguments.
 */
EXPORTCH double doublesum(double a, double b)
{
	double retval;
	printf("In sum: a = %lf, b = %lf\n", a, b);
	retval = a + b;
	printf("Returning %lf\n", retval);
	fflush(stdout);
	return retval;
}

/**
 * \brief     Structures by value example.
 * 
 * This function demonstrates passing structures
 * by value. The two structs uses counters are 
 * incremented, though the arguments to the function
 * will not change due to be passed by value.
 * 
 * \param	  s1 A struct1_t passed by value
 * \param	  s2 A struct2_t passed by value
 * \returns   The sum of the two struct's i fields
 */
EXPORTCH int func1(struct1_t s1, struct2_t s2)
{
	int retval;
	printf("In func1: s1.i = %i, s2.i = %i\n", s1.i, s2.i);
	retval = s1.i + s2.i;
	printf("Returning %i\n", retval);

	s1.uses++;
	s2.uses++;

	fflush(stdout);
	return retval;
}

/**
 * \brief     Structures by reference example.
 * 
 * This function demonstrates passing structures
 * by reference. The two structs uses counters are 
 * incremented. The counters in the argument will 
 * change if the code is called from another C function.
 * However, if called from C#, the values will not 
 * change unless the arguments are declared as "ref"
 * or they are explicitly marshaled.
 * 
 * \param	  s1 A struct1_t pointer
 * \param	  s2 A struct2_t pointer
 * \returns   The sum of the two struct's i fields
 */
EXPORTCH double func2(struct1_t *s1, struct2_t *s2)
{
	double retval;
	printf("In func1: s1->d = %lf, s2->d = %lf\n", s1->d, s2->d);
	retval = s1->d + s2->d;
	printf("Returning %lf\n", retval);

	s1->uses++;
	s2->uses++;

	fflush(stdout);
	return retval;
}

/**
 * \brief     Returning a struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new struct2_s, s2, is allocated. 
 * The values of s1 are copied to s2, and the 
 * struct1_s pointer in the new s2 is set to s1.
 * 
 * \param	  s1 A struct1_t pointer
 * \returns   A new struct2_t copied from s1
 */
EXPORTCH struct2_t* func3(struct1_t* s1)
{
	if (s1 != NULL)
	{
		struct2_t* s2 = (struct2_t*) malloc(sizeof(struct2_t));
		s2->d = s1->d;
		s2->i = s1->i;
		s2->uses = s1->uses;
		memcpy(s2->iarr, s1->iarr, sizeof(s1->iarr));
		s2->s = s1;
		if (s1->name != NULL)
		{
			s2->name = (char*)malloc(strlen(s1->name) + 1);
			strcpy_s(s2->name, strlen(s1->name) + 1, s1->name);
		}
		else
		{
			s2->name = NULL;
		}

		s1->uses++;

		return s2;
	}
	else
		return NULL;
}

/**
 * \brief     Returning a struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new struct1_s, s, is allocated. 
 * The values of s are set to defaults.
 * 
 * \returns   The sum of the two struct's i fields
 */
EXPORTCH struct1_t* new_struct1(void)
{
	struct1_t* s = (struct1_t*) malloc(sizeof(struct1_t));
	if (s != NULL)
	{
		count++;
		s->d = 3.14;
		s->i = 314;
		s->uses = -1;
		s->name = n1;
		memset(s->iarr, -1, sizeof(s->iarr));
	}
	return s;
}

/**
 * \brief     Frees a dynamically allocated struct1_s
 * 
 * This function frees a dynamically allocated struct1_s.
 * It can be used to simplify interaction from C#, but is
 * not strictly necessary.
 * 
 * \param     s A pointer to a struct1_s
 */
EXPORTCH void free_struct1(struct1_t* s)
{
	if (s != NULL)
		free(s);
}

/**
 * \brief     Frees a dynamically allocated struct2_s
 * 
 * This function frees a dynamically allocated struct2_s.
 * It can be used to simplify interaction from C#, but is
 * not strictly necessary.
 * 
 * \param     s A pointer to a struct2_s
 */
EXPORTCH void free_struct2(struct2_t* s)
{
	if (s != NULL)
		free(s);
}
