using System;
using System.Collections.Generic;
using System.Text;
using LibMC;
using System.Threading;
using System.Runtime.InteropServices;

/** \file /LibMCInterop/Program.cs
 * Program for starting a Mobile-C agency from a console and
 * interacting with an agent.
 */

/**
 * \brief      Namespace for the LibMCInterop test program.
 * 
 * This program demonstrates how to start an agency, find an agent,
 * use the Mobile-C command line. Also demonstrates how to terminate
 * an agent.
 */
namespace LibMCInterop
{
    /**
     * \brief     Class for the main program.
     * 
     * Contains the Main function and variables, structs, etc. used
     * by the program.
     */
    class Program
    {
        public static Cs2Ch.Fx d;

        public static MCAgency Agency = new MCAgency();
        public static int localport = 5050;
        public static int remoteport = 5051;

        /**
         * \brief     Main program for a general console program.
         * 
         * Creates an agency and hosts agents. It also looks for an agent
         * named "persistent1" and attempts to call functions in that agent
         * script file. Demonstrates calling agent functions, terminating
         * an agent, waiting on agency signals, and importing structures
         * from a script file.
         */
        static void Main(string[] args)
        {
            int temp;
            MCAgent agent;

            // Calling a library function
            Console.WriteLine("get_inc_int(): " + Cs2Ch.get_inc_int().ToString());
            Console.WriteLine("add(): " + Cs2Ch.add(14, 434).ToString());

            /* Setting up the callbacks
             * This must be done before the agency is started
             * so that the library is loaded and ready with the
             * callback pointer before any agent tries to use the 
             * callback.
             */
            d = new Cs2Ch.Fx(WriteLine);
            Cs2Ch.SetCallBack(d);
            Cs2Ch.callFptr("This is C# calling back to itself.");

            // Starting the agency
            Console.WriteLine("Starting Mobile-C on port " +
                localport.ToString() + ".\n");

            Agency.Port = localport;
            temp = Agency.Initialize();
            if (temp != 0)
                Console.WriteLine("Initialize: " + temp.ToString() + "\n");

            while (true)
            {
                // Wait for an agent to arrive and execute
                Agency.WaitSignal((int)MCAgency.MC_Signal_e.MC_EXEC_AGENT);
                Agency.ResetSignal();
                try
                {
                    // Find the persistent agent
                    agent = Agency.FindAgentByName("persistent1");
                    Console.WriteLine("Found agent\n" + agent.ToString() + ".");
                    while (agent.AgentStatus != MCAgent.MC_AgentStatus_e.MC_AGENT_NEUTRAL)
                    {
                        // Agent is not in the right state - wait
                        Console.WriteLine("Agent is not yet ready.");
                        Thread.Sleep(250);
                    }

                    Console.WriteLine("Attempting to call simple agent function func().");
                    temp = agent.CallAgentFunc("func", IntPtr.Zero, IntPtr.Zero);
                    Console.WriteLine("CallAgentFunc() for func() returned " +
                        temp.ToString() + ".");

                    Cs2Ch.arg_struct arg = new Cs2Ch.arg_struct();
                    arg.a = 11;
                    arg.b = 43;
                    int i = 0;

                    Console.WriteLine("Attempting to call complex " +
                        "agent function hello().");
                    Console.WriteLine("Before function call:");
                    Console.WriteLine("  arg.a = " + arg.a.ToString());
                    Console.WriteLine("  arg.b = " + arg.b.ToString());

                    // Box the arguments - see documentation for more details
                    object a = (object)i, b = (object)arg;
                    temp = agent.CallAgentFunc("hello", ref a, ref b);
                    // Unbox the arguments.
                    i = (int)a;
                    arg = (Cs2Ch.arg_struct)b;

                    Console.WriteLine("CallAgentFunc() returned " +
                        temp.ToString() + ".");
                    Console.WriteLine("Agent function hello() returned " +
                        i.ToString() + ".");
                    Console.WriteLine("After function call:");
                    Console.WriteLine("  arg.a = " + arg.a.ToString());
                    Console.WriteLine("  arg.b = " + arg.b.ToString());
                }
                catch (InvalidAgentException)
                {
                    // This will catch an exception caused by 
                    // persistent1 not being found.
                    Console.WriteLine("Did not find the agent.");
                }
                catch (ArgumentException e)
                {
                    // This will catch an exception caused by 
                    // CallAgentFunc casting or another error.
                    Console.WriteLine("Argument Exception: " + e.Message);
                }
                catch (OutOfMemoryException e)
                {
                    // This will catch an exception caused by 
                    // CallAgentFunc failing to allocate memory.
                    Console.WriteLine("Memory Exception: " + e.Message);
                }
            }
        }

        /**
         * \brief     Delegate function for agent callback.
         * 
         * This function is used to create the delegate that
         * the cs2ch library uses to allow agents to call back
         * to managed code.
         */
        public static int WriteLine(String s)
        {
            Console.WriteLine(s);
            return 123;
        }
    }
}
