using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

/** \file /LibMCInterop/Cs2Ch.cs
 * Class file for importing functions from cs2ch.dl.
 */

namespace LibMCInterop
{
    /**
     * \brief     Wrapper class for the example agent interface.
     * 
     * Contains structure definitions, imports from DL file, and
     * wrapper functions for imported functions.
     */
    public class Cs2Ch
    {
        /**
         * \brief     Path to the dl file built by the cs2ch example
         * 
         * If you have not installed Ch to the default directory, 
         * your main hard drive is not "C:", or you have installed
         * cs2ch in a directory other than the default, you will need 
         * to change this to make the project work correctly.
         */
        private const String dlfile =
            "C:\\Ch\\toolkit\\embedch\\package\\cs2ch\\dl\\libcs2ch.dl";    // Default install location
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\Debug\\libcs2ch.dl"; // Debug build from VS
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\Release\\libcs2ch.dl"; // Release build from VS
            //"C:\\mobilec\\demos\\win32\\LibMC.Net\\Ch\\cs2ch\\cs2ch\\dl\\libcs2ch.dl"; // dlcomp/dllink build from package

        /**
         * \brief     Delegate function for callback in cs2ch.dl
         * 
         * This is the delegate function for the callback used by 
         * setCallback() and callFptr() in cs2ch.dl. It is the rough 
         * equivalent of "typedef int (__stdcall *Fx)(char*)". See the
         * documentation for interoperability for more information.
         */
        public delegate int Fx(String s);

        /**
         * \brief     Argument structure for agent function "hello."
         * 
         * Defined in the persistent.xml agent example.
         */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        public struct arg_struct
        {
            public int a;
            public int b;
        }

        /**
         * \brief     A structure from cs2ch.dl
         * 
         * Includes an int, double, fixed length array, and 
         * a String.
         */
        [StructLayout(LayoutKind.Sequential)]
        public struct struct1_t
        {
            public int i;
            public double d;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public int[] iarr;
            public int uses;
            public String name;
        };

        /**
         * \brief     A structure from cs2ch.dl
         * 
         * Includes an int, double, fixed length array, and 
         * a String, and a pointer to another structure.
         */
        [StructLayout(LayoutKind.Sequential)]
        public struct struct2_t
        {
            public int i;
            public double d;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public int[] iarr;
            public IntPtr s; // pointer to struct struct1_t
            public int uses;
            public String name;
        };

        /**
         * \brief     Sets the callback in the library.
         * 
         * Sets the callback in the cs2ch library. Handles 
         * creating the function pointer from the delegate
         * automatically.
         * 
         * \param     callback Fx delegate to use as callback
         */
        public static void SetCallBack(Fx callback)
        {
            IntPtr f = Marshal.GetFunctionPointerForDelegate(callback);
            Cs2Ch.setCallback(f);
        }

        public static struct1_t NewStruct1()
        {
            IntPtr sp = new_struct1();
            struct1_t s = (struct1_t)Marshal.PtrToStructure(sp, typeof(struct1_t));
            free_struct1(sp);
            return s;
        }

        public static struct2_t Func3(ref struct1_t s1)
        {
            IntPtr sp = func3(ref s1);
            if (sp != IntPtr.Zero)
            {
                struct2_t s = (struct2_t)Marshal.PtrToStructure(sp, typeof(struct2_t));
                free_struct2(sp);
                return s;
            }
            else
                return new struct2_t();
        }

        [DllImport(dlfile)]
        public static extern void dll_init();
        [DllImport(dlfile)]
        public static extern void dll_exit();
        [DllImport(dlfile)]
        public static extern int get_inc_int();
        [DllImport(dlfile)]
        public static extern int get_int();
        [DllImport(dlfile)]
        public static extern int add(int a, int b);
        [DllImport(dlfile)]
        public static extern double doublesum(double a, double b);
        [DllImport(dlfile)]
        public static extern int func1([MarshalAs(UnmanagedType.Struct)] struct1_t s1, [MarshalAs(UnmanagedType.Struct)] struct2_t s2);
        [DllImport(dlfile)]
        public static extern double func2(ref struct1_t s1, ref struct2_t s2);
        [DllImport(dlfile)]
        public static extern void callFptr(String s);

        // These functions can be called directly, but wrappers are provided as well.
        [DllImport(dlfile)]
        public static extern void setCallback(IntPtr f);

        // These functions require manual marshalling or involve unmanaged dynamic memory, so they will be wrapped
        [DllImport(dlfile)]
        internal static extern IntPtr func3(ref struct1_t s1);
        [DllImport(dlfile)]
        internal static extern IntPtr new_struct1();
        [DllImport(dlfile)]
        internal static extern void free_struct1(IntPtr s);
        [DllImport(dlfile)]
        internal static extern void free_struct2(IntPtr s);

    }
}
