/**
* \file cs2chfunc.c
*
* This file contains function definitions for the cs2ch
* library. This library is designed to show how to allow
* agents and/or C code to interact with C# and other CLI
* languages. See the individual functions for their 
* descriptions and purposes.
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "cs2ch.h"    /* header file for cs2ch lib */

/* Macro for safe string copy under Windows */
#ifdef _CH_
#define strcpy_s(dest, len, src) strcpy(dest, src)
#define _strdup(str) strdup(str)
#endif

// A function pointer for the callback.
Fx fptr = NULL;
Fy func2ptr = NULL;

// Static variables to show persistence of the library in memory.
int count = 0;
struct1_t *dll_s1 = NULL;
struct2_t *dll_s2 = NULL;
char* n1 = "Name 1";
char* n2 = "Name 2";

/* This is an optional function for Windows only that 
   allows you to specify DLL entry and exit routines */
/*
#ifdef _WIN32
#include <windows.h>
BOOL WINAPI DllMain(
  HANDLE hinstDLL, 
  DWORD dwReason, 
  LPVOID lpvReserved
)
{
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);
	printf("Dll main running.\n");
	fflush(stdout);
    count = 0;
	return TRUE; // Return FALSE to abort loading library
}
#endif
*/

/**
 * \brief     Sets an Fx callback pointer in the library.
 * 
 * Calls to this library can use the callback pointer
 * to call any function with the appropriate signature.
 * This function demonstrates how to set a callback pointer
 * from C or C#.
 *
 * \note      The function pointer is checked for NULL.
 * \sa		  callFptr
 *
 * \param	  f An Fx callback pointer.
 * \returns   1 if the callback is assigned, 0 otherwise.
 */
EXPORTCH int setFxCallback(Fx f)
{
	if (f != NULL)
	{
		fptr = f;
        return TRUE;
	}
    else
    {
        return FALSE;
    }
}

/**
 * \brief     Calls the Fx callback pointer in the library.
 * 
 * This function is used to call the Fx callback pointer 
 * stored in the library. To the calling code, it has the
 * appearance of simply calling a function with the same 
 * Fx prototype, but the arguments and return values are
 * passed to and from the underlying Fx function pointer.
 *
 * \note      The function pointer is checked for NULL.
 * \sa		  setCallback
 *
 * \param	  s A string to pass to the callback
 * \returns   The return value of the callback function
 */
EXPORTCH int callFxCallback(char* s)
{
	int i = 0;
	if (fptr != NULL)
	{
        if (s != NULL)
        {
            i = (*fptr)(s);
            printf("Returned %i\n", i);
            fflush(stdout);
        }
	}
    else
    {
        fprintf(stderr, "Callback function pointer not set!\n");
        fflush(stderr);
    }
    return i;
}

/**
 * \brief     Sets an Fy callback pointer in the library.
 * 
 * Calls to this library can use the callback pointer
 * to call any function with the appropriate signature.
 * This function demonstrates how to set a callback pointer
 * from C or C#.
 *
 * \note      The function pointer is checked for NULL.
 * \sa		  callFunc2
 *
 * \param	  f An Fy callback pointer.
 * \returns   1 if the callback is assigned, 0 otherwise.
 */
EXPORTCH int setFyCallback(Fy f)
{
	if (f != NULL)
	{
		func2ptr = f;
        return TRUE;
	}
    else
    {
        return FALSE;
    }
}

/**
 * \brief     Calls the Fy callback pointer in the library.
 * 
 * This function is used to call the Fy callback pointer 
 * stored in the library. To the calling code, it has the
 * appearance of simply calling a function with the same 
 * Fy prototype, but the arguments and return values are
 * passed to and from the underlying Fx function pointer.
 *
 * \note      The function pointer is checked for NULL.
 * \sa		  setFunc2Callback
 *
 * \param	  s An arg_struct pointer to pass to the callback
 * \returns   The return value of the callback function
 */
EXPORTCH int callFyCallback(arg_struct* s)
{
	int i = 0;
	if (func2ptr != NULL)
	{
        if (s != NULL)
        {
            i = (*func2ptr)(s);
            printf("Returned %i\n", i);
            fflush(stdout);
        }
	}
    else
    {
        fprintf(stderr, "Callback function pointer not set!\n");
        fflush(stderr);
    }
    return i;
}

/**
 * \brief     Returns and increments a static int.
 * 
 * This function returns the value of an int stored
 * in the library and increments it. It can be used to 
 * show persistence in the library and can be called
 * directly from any language.
 *
 * \returns   The (unincremented) value of the static int
 */
EXPORTCH int get_inc_int(void)
{
    printf("Count = %i.\n", count);
    fflush(stdout);
    return count++;
}

/**
 * \brief     Returns the value of a static int.
 * 
 * This function returns the value of an int stored
 * in the library. It can be used to 
 * show persistence in the library and can be called
 * directly from any language.
 *
 * \returns   The value of the static int
 */
EXPORTCH int get_int(void)
{
    printf("Count = %i.\n", count);
    fflush(stdout);
    return count;
}

EXPORTCH int dll_init(void)
{
	count = 0;

	dll_s1 = (struct1_t*) malloc(sizeof(struct1_t));
	dll_s1->d = 0;
	dll_s1->i = 0;
	dll_s1->uses = -1;
	dll_s1->name = _strdup(n1);
	memset(dll_s1->iarr, -1, sizeof(dll_s1->iarr));

	dll_s2 = (struct2_t*) malloc(sizeof(struct2_t));
	dll_s2->d = 3.14159;
	dll_s2->i = 314159;
	dll_s2->uses = 0;
	dll_s2->s = dll_s1;
	dll_s2->name = _strdup(n2);
	memset(dll_s2->iarr, -1, sizeof(dll_s2->iarr));
    
    return count;
}

EXPORTCH int dll_exit(void)
{
	if (dll_s1 != NULL)
		free(dll_s1);
	if (dll_s2 != NULL)
		free(dll_s2);
    return count;
}

/**
 * \brief     Simple function that adds two ints.
 * 
 * This function adds two ints and returns the
 * result. It can be called from any language, but
 * care must be taken that the size of the integer
 * argument is the same across all platforms.
 *
 * \returns   The sum of the arguments.
 */
EXPORTCH int add(int a, int b)
{
	int retval;
	retval = a + b;
	printf("In sum: a = %i, b = %i\n", a, b);
	printf("Returning %i\n", retval);
	fflush(stdout);

	fprintf(stderr, "This is an error.\n");
	fflush(stderr);

	return retval;
}

/**
 * \brief     Simple function that adds to doubles.
 * 
 * This function adds two doubles returns the
 * result. It can be called from any language, but
 * care must be taken that the size of the double
 * argument is the same across all platforms.
 *
 * \returns   The sum of the arguments.
 */
EXPORTCH double doublesum(double a, double b)
{
	double retval;
	printf("In sum: a = %lf, b = %lf\n", a, b);
	retval = a + b;
	printf("Returning %lf\n", retval);
	fflush(stdout);
	return retval;
}

/**
 * \brief     Structures by value example.
 * 
 * This function demonstrates passing structures
 * by value. The two structs uses counters are 
 * incremented, though the arguments to the function
 * will not change due to being passed by value.
 *
 * \note      This function can be used to show how
 * automatic marshalling takes place between C and C#
 * with structs passed by value. While the function
 * increments both uses fields, the structs in the 
 * calling space will not be changed.
 * 
 * \param	  s1 A struct1_t passed by value
 * \param	  s2 A struct2_t passed by value
 * \returns   The sum of the two structs' i fields
 */
EXPORTCH int func1(struct1_t s1, struct2_t s2)
{
	int retval;
	printf("In func1: s1.i = %i, s2.i = %i\n", s1.i, s2.i);
	retval = s1.i + s2.i;
	printf("Returning %i\n", retval);

	s1.uses++;
	s2.uses++;

	fflush(stdout);
	return retval;
}

/**
 * \brief     Structures by reference example.
 * 
 * This function demonstrates passing structures
 * by reference. The two structs uses counters are 
 * incremented. The counters in the argument will 
 * change if the code is called from another C function.
 * However, if called from C#, the values will not 
 * change unless the arguments are declared as "ref"
 * or they are explicitly marshaled.
 *
 * \note	  The function increments the uses field
 * of both structs. If declared properly in C#, using
 * the "ref" keyword, marshalling will automatically 
 * copy the arguments into and out of the function call,
 * allowing this function to effectively modify the 
 * structs in C# space.
 * 
 * \param	  s1 A struct1_t pointer
 * \param	  s2 A struct2_t pointer
 * \returns   The sum of the two struct's i fields
 */
EXPORTCH double func2(struct1_t *s1, struct2_t *s2)
{
	double retval;
	printf("In func1: s1->d = %lf, s2->d = %lf\n", s1->d, s2->d);
	retval = s1->d + s2->d;
	printf("Returning %lf\n", retval);

	s1->uses++;
	s2->uses++;

	fflush(stdout);
	return retval;
}

/**
 * \brief     Returning a struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new struct2_s, s2, is allocated. 
 * The values of s1 are copied to s2, and the 
 * struct1_s pointer in the new s2 is set to s1.
 *
 * \note	  The argument uses field is incremented,
 * and will be reflected in C# space if the prototype
 * is properly declared. The returned value cannot be
 * marshaled automatically. Explicit marshaling must copy
 * the returned struct into a C#-space struct and deallocate
 * the memory allocated by the function.
 * 
 * \param	  s1 A struct1_t pointer
 * \returns   A new struct2_t copied from s1
 */
EXPORTCH struct2_t* func3(struct1_t* s1)
{
	if (s1 != NULL)
	{
		struct2_t* s2 = (struct2_t*) malloc(sizeof(struct2_t));
		s2->d = s1->d;
		s2->i = s1->i;
		s2->uses = s1->uses;
		memcpy(s2->iarr, s1->iarr, sizeof(s1->iarr));

		/* Here is an assignment that requires attention.
		 * The argument s1 is expected to be a pointer to
		 * a struct that is available to this C library.
		 * However, if s1 is automatically marshalled by
		 * P-Invoke, it is not a valid pointer that can
		 * be retained. Therefore, s1 must be copied
		 * so that s2 can retain a pointer to s1.
		 */
		s2->s = (struct1_t*) malloc(sizeof(struct1_t));
		if (s2->s != NULL)
		{
			count++;
			s2->s->d = s1->d;
			s2->s->i = s1->i;
			s2->s->uses = s1->uses;
			s2->s->name = _strdup(s1->name);
			memcpy(s2->s->iarr, s1->iarr, sizeof(s1->iarr));
		}

		if (s1->name != NULL)
		{
			s2->name = _strdup(s1->name);
			//(char*)malloc(strlen(s1->name) + 1);
			//strcpy_s(s2->name, strlen(s1->name) + 1, s1->name);
		}
		else
		{
			s2->name = NULL;
		}

		s1->uses++;

		return s2;
	}
	else
		return NULL;
}

/**
 * \brief     Returning a struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new struct2_s, s2, is allocated. 
 * The values of s1 are copied to s2, and the 
 * struct1_s pointer in the new s2 is set to NULL.
 *
 * \note	  The argument uses field is incremented,
 * and but will not be changed in C# space because the s1
 * is passed by value. The returned value cannot be
 * marshaled automatically. Explicit marshaling must copy
 * the returned struct into a C#-space struct and deallocate
 * the memory allocated by the function.
 * 
 * \param	  s1 A struct1_t
 * \returns   A new struct2_t copied from s1
 */
EXPORTCH struct2_t* func4(struct1_t s1)
{
	struct2_t* s2 = (struct2_t*) malloc(sizeof(struct2_t));
	s2->d = s1.d;
	s2->i = s1.i;
	s2->uses = s1.uses;
	memcpy(s2->iarr, s1.iarr, sizeof(s1.iarr));
	s2->s = NULL;
	if (s1.name != NULL)
	{
		s2->name = _strdup(s1.name);
		//s2->name = (char*)malloc(strlen(s1.name) + 1);
		//strcpy_s(s2->name, strlen(s1.name) + 1, s1.name);
	}
	else
	{
		s2->name = NULL;
	}

	s1.uses++;

	return s2;
}

/**
 * \brief     Returning a struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new struct1_s, s, is allocated. 
 * The values of s are set to defaults.
 *
 * \note	  The returned value cannot be
 * marshaled automatically. Explicit marshaling must copy
 * the returned struct into a C#-space struct and deallocate
 * the memory allocated by the function.
 * 
 * \returns   A new struct1_s
 */
EXPORTCH struct1_t* new_struct1(void)
{
	struct1_t* s = (struct1_t*) malloc(sizeof(struct1_t));
	if (s != NULL)
	{
		count++;
		s->d = 3.14;
		s->i = 314;
		s->uses = -1;
		s->name = _strdup(n1);
		memset(s->iarr, -1, sizeof(s->iarr));
	}
	return s;
}

/**
 * \brief     Returning a simple struct example
 * 
 * This function demonstrates returning a structure
 * in dynamic memory. A new arg_struct, s, is allocated. 
 * The values of s are set to defaults.
 * 
 * \returns   A new arg_struct with values 55 and 99
 */
EXPORTCH arg_struct* new_arg_struct(void)
{
	arg_struct* s = (arg_struct*) malloc(sizeof(arg_struct));
	if (s != NULL)
	{
		s->a = 55;
		s->b = 99;
	}
	return s;
}

/**
 * \brief     Frees a dynamically allocated struct1_s
 * 
 * This function frees a dynamically allocated struct1_s.
 * It can be used to simplify interaction from C#, but is
 * not strictly necessary.
 *
 * \note	  This function can be called from C# to free
 * a struct returned by new_struct1. Alternatively, the 
 * memory returned by new_struct1 can be freed directly 
 * by the C# API, but this handles freeing all the members
 * as well.
 * 
 * \param     s A pointer to a struct1_s
 */
EXPORTCH void free_struct1(struct1_t* s)
{
	if (s != NULL)
		if (s->name != NULL)
			free(s->name);
		free(s);
}

/**
 * \brief     Frees a dynamically allocated struct2_s
 * 
 * This function frees a dynamically allocated struct2_s.
 * It can be used to simplify interaction from C#, but is
 * not strictly necessary.
 *
 * \note	  This function can be called from C# to free
 * a struct returned by func3. Alternatively, the 
 * memory returned by func3 can be freed directly
 * by the C# API, but this handles freeing all the members
 * as well.
 * 
 * \param     s A pointer to a struct2_s
 */
EXPORTCH void free_struct2(struct2_t* s)
{
	if (s != NULL)
	{
		free_struct1(s->s);
		if (s->name != NULL)
			free(s->name);
		free(s);
	}
}
