using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

/**
 * \page mcagent MCAgent
 * Examples of commonly used MCAgent operations:
 * Find an agent by name (assumes an MCAgency named Agency):
 * \code
 * MCAgent agent;
 * try
 * {
 *     agent = Agency.FindAgentByName("persistent1");
 * }
 * catch (Exception e)
 * {
 *     Console.WriteLine("Exception: " + e.Message);
 * }
 * \endcode
 *
 * Terminate an agent
 * \code
 * int temp;
 * try
 * {
 *     temp = agent.TerminateAgent();
 *     Console.WriteLine("TerminateAgent() returned " + 
 *         temp.ToString() + ".");
 * }
 * catch (Exception e)
 * {
 *     Console.WriteLine("Exception: " + e.Message);
 * }
 * 
 * \endcode
 * 
 * Print information about an agent:
 * \code
 * Console.WriteLine(agent.ToString());
 * Console.WriteLine(agent.GetAgentXMLString());
 * Console.WriteLine(agent.RetrieveAgentCode());
 * \endcode
 */

/** \file MCAgent.cs
 * Defines the MCAgent object and its member functions.
 */

namespace LibMC
{
    /**
     * \brief   Wrapper class for MCAgent_t structure.
     * 
     * This class provides an interface to the Mobile-C agent structure.
     * Member functions for the class are generally overloaded versions of the 
     * respective functions in the Mobile-C library. The class maintains a 
     * pointer to a Mobile-C agent in unmanaged memory. The pointer
     * is not accessible by the user.
     */
    public class MCAgent
    {
        private IntPtr agent_p;
        private String name = "";
        private int id = -1;
        private int numTasks = -1;
        private MC_AgentStatus_e status = MC_AgentStatus_e.MC_NO_STATUS;
        private MC_AgentType_e type = MC_AgentType_e.MC_NONE;

        /**
         * \brief   Enum for describing the type of an agent.
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         */
        public enum MC_AgentType_e
        {
            MC_NONE = -1,           /**< Default value to describe unininitialized agent. */
            MC_REMOTE_AGENT = 0,    /**< A remote agent. */
            MC_LOCAL_AGENT,         /**< A local agent. */
            MC_RETURN_AGENT         /**< A returning agent. */
        };

        /**
         * \brief   Enum for describing the status of an agent.
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         */
        public enum MC_AgentStatus_e
        {
            MC_NO_STATUS = -1, /*!< Default value for uninitialized agent */
            MC_WAIT_CH = 0,    /*!< Waiting to be started */
            MC_WAIT_MESSGSEND, /*!< Finished, waiting to migrate */
            MC_AGENT_ACTIVE,   /*!< Running */
            MC_AGENT_NEUTRAL,  /*!< Not running, but do not flush */
            MC_AGENT_SUSPENDED,/*!< Unused */
            MC_WAIT_FINISHED   /*!< Finished, waiting to be flushed */
        };

        /**
         * \brief       Default constructor
         * 
         * Creates an empty agent.
         */
        public MCAgent()
        {
            // Not using wrapper - default value
            agent_p = IntPtr.Zero;
        }

        internal MCAgent(IntPtr ip)
        {
            Agent = ip;
        }

        /**
         * \brief       Display the agent's fields.
         * 
         * Formats an returns a string with all of the agents properties.
         * 
         * \returns     A string containing a formatted representation of
         *              the agent's properties.
         * 
         * \note        The agency port and any other options must be set
         *              before calling this function.
         */
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("MCAgent:\n\tName: {0}\n\tID: {1}\n\tNumTasks: {2}\n\tStatus: {3}\n\tType: {4}", 
                AgentName.ToString(), 
                AgentID.ToString(),
                AgentNumTasks.ToString(),
                Enum.GetName(typeof(MC_AgentStatus_e), AgentStatus),
                Enum.GetName(typeof(MC_AgentType_e), AgentType));

            return sb.ToString();
        }

        /*
         * Accessors
         */
        internal IntPtr Agent
        {
            get
            {
                if (agent_p == IntPtr.Zero)
                    throw new InvalidAgentException();
                else
                    return agent_p;
            }
            set
            {
                if (agent_p == IntPtr.Zero)
                {
                    agent_p = value;
                    GetAgentFields();
                }
                else
                    throw new InvalidAgentException("Attempting to assign new agent to non-zero agent pointer!");
            }
        }

        private void GetAgentFields()
        {
            id = MCAgency._MC_GetAgentID(Agent);
            name = MCAgency._MC_GetAgentName(Agent);
            numTasks = MCAgency._MC_GetAgentNumTasks(Agent);
            status = MCAgency._MC_GetAgentStatus(Agent);
            type = MCAgency._MC_GetAgentType(Agent);
        }

        /**
         * \brief       Gets the agent's ID number.
         * 
         * Gets the agent's ID number as assigned by Mobile-C
         * if the agent is a valid agent.
         * 
         * \returns     The agent's ID number or -1 for an empty agent.
         */
        public int AgentID
        {
            get
            {
                return id;
            }
            /*set
            {
                id = value;
            }*/
        }

        /**
         * \brief       Gets the agent's name.
         * 
         * Gets the agent's name as assigned by Mobile-C or the
         * agent script if the agent is a valid agent.
         * 
         * \returns     The agent's name or an empty string for an empty agent.
         */
        public String AgentName
        {
            get
            {
                return name;
            }
            /*set
            {
                name = value;
            }*/
        }

        /**
         * \brief       Gets the agent's number of tasks.
         * 
         * Gets the agent's ID number of tasks if the agent is a valid agent.
         * 
         * \returns     The agent's ID number of tasks or -1 for an empty agent.
         */
        public int AgentNumTasks
        {
            get
            {
                return numTasks;
            }
            /*set
            {
                numTasks = value;
            }*/
        }

        /**
         * \brief       Gets or sets the agent's status.
         * 
         * Gets or sets the agent's status. When setting the status,
         * the status is double-checked after setting it and may not
         * be set depending on the state of the agent and the agency.
         * 
         * \returns     The agent's status or MC_NO_STATUS for an
         *              empty agent.
         */
        public MC_AgentStatus_e AgentStatus
        {
            get
            {
                status = MCAgency._MC_GetAgentStatus(Agent);
                return status;
            }
            set
            {
                MCAgency._MC_SetAgentStatus(Agent, value);
                status = MCAgency._MC_GetAgentStatus(Agent);
            }
        }

        /**
         * \brief       Gets the agent's type.
         * 
         * Gets the agent's type.
         * 
         * \returns     The agent's type or MC_NONE for an
         *              empty agent.
         */
        public MC_AgentType_e AgentType
        {
            get
            {
                return type;
            }
            /*set
            {
                type = value;
            }*/
        }

        /**
         * \brief       Checks whether the agent is valid.
         * 
         * Checks the internal agent pointer to see if it is
         * non-zero.
         * 
         * \returns     True if the pointer is valid, false
         *              otherwise
         */
        public bool Valid
        {
            get
            {
                return (agent_p != IntPtr.Zero);
            }
            /*set
            {
                type = value;
            }*/
        }

        /*
         * Static casting operators
         */
        static public implicit operator IntPtr(MCAgent agent)
        {
            return agent.Agent;
        }

        static public implicit operator MCAgent(IntPtr ip)
        {
            return new MCAgent(ip);
        }

        /*
         * Agent functions
         */

        /**
         * \brief       Deletes an agent.
         * 
         * Deletes an agent from the agency. 
         * 
         * \returns     The return value of the underlying MC_DeleteAgent
         *              function call.
         */
        public int DeleteAgent()
        {
            return MCAgency._MC_DeleteAgent(Agent);
        }

        /**
         * \brief       Gets the agent's XML string.
         * 
         * Returns the full XML string associated with the agent.
         * 
         * \returns     The return value of the underlying MC_GetAgentXMLString
         *              function call.
         */
        public String GetAgentXMLString()
        {
            return MCAgency._MC_GetAgentXMLString(Agent);
        }

        /**
         * \brief       Gets the agent's C code string.
         * 
         * Prints the C code associated with the agent to stdout.
         * 
         * \returns     The return value of the underlying MC_PrintAgentCode
         *              function call.
         */
        public int PrintAgentCode()
        {
            return MCAgency._MC_PrintAgentCode(Agent);
        }

        /**
         * \brief       Gets the agent's C code string.
         * 
         * Returns the C code associated with the agent.
         * 
         * \returns     A string containing the agent's C code.
         */
        public String RetrieveAgentCode()
        {
            return MCAgency._MC_RetrieveAgentCode(Agent);
        }

        /**
         * \brief       Terminates an agent.
         * 
         * Terminates an agent regardless of the agent's state.
         * 
         * \returns     The return value of the underlying MC_TerminateAgent
         *              function call.
         */
        public int TerminateAgent()
        {
            return MCAgency._MC_TerminateAgent(Agent);
        }

        /*public int _MC_CallAgentFunc(IntPtr agent, String funcName, void* returnVal, void* varg);*/
        /*public void* _MC_GetAgentExecEngine(IntPtr agent);*/
        /*public int _MC_GetAgentReturnData(IntPtr agent, int task_num, void **data, int *dim, int **extent);*/

        /*
         * ACL Functions
         */

        /**
         * \brief       Posts an ACL message to the agent.
         * 
         * Delivers an ACL message to the agent.
         * 
         * \param       message The ACL message object to deliver.
         * \returns     The return value of the underlying MC_AclPost
         *              function call.
         * 
         * \note        The message must be a valid message or this
         *              function call will fail.
         */
        public int AclPost(MCAclMessage message)
        {
            return MCAgency._MC_AclPost(Agent, message.AclMsg);
        }

        /**
         * \brief       Retrieve an ACL message from the agent.
         * 
         * Retrieves an ACL message from the agent if one is available.
         * 
         * \returns     The ACL message or a blank ACL message if one
         *              was not available.
         * 
         * \note        The message must be a valid message or this
         *              function call will fail.
         */
        public MCAclMessage AclRetrieve()
        {
            IntPtr temp = MCAgency._MC_AclRetrieve(Agent);
            if (temp == IntPtr.Zero)
                return new MCAclMessage();
            else
                return new MCAclMessage(temp);
        }

        /**
         * \brief       Wait for and retrieve an ACL message from the agent.
         * 
         * Retrieves an ACL message from the agent when one becomes available.
         * 
         * \returns     The ACL message or a blank ACL message if the call
         *              fails.
         * 
         * \note        This function call blocks.
         */
        public MCAclMessage AclWaitRetrieve()
        {
            IntPtr temp = MCAgency._MC_AclWaitRetrieve(Agent);
            if (temp == IntPtr.Zero)
                return new MCAclMessage();
            else
                return new MCAclMessage(temp);
        }

        /**
         * \brief       Calls a function in an agent script.
         * 
         * Calls a function in an agent's script file. This function
         * requires manual marshaling by the user.
         * 
         * \param       funcName The name of the function to call
         * \param       retval A pointer to memory for the return value
         * \param       varg A pointer to the argument for the function
         * \returns     The return value of the underlying MC_CallAgentFunc
         *              function call.
         * 
         * \note        BE VERY CAREFUL! You must marshal your arguments!
         *              If possible, use the other CallAgentFunc that handles
         *              marshaling automatically.
         * \sa          Overloaded CallAgentFunc, LibMCConsole example
         */
        public int CallAgentFunc(String funcName, IntPtr retval, IntPtr varg)
        {
            return MCAgency._MC_CallAgentFunc(Agent, funcName, retval, varg);
        }

        /**
         * \brief       Calls a function in an agent script.
         * 
         * Calls a function in an agent's script file. This function
         * requires boxing of parameters, but no marshaling.
         * 
         * \param       funcName The name of the function to call
         * \param       retval A boxed object to hold the return value
         * \param       varg The boxed argument to the agent function
         * \returns     The return value of the underlying MC_CallAgentFunc
         *              function call.
         * 
         * \note        This function handles marshaling of the argument
         *              and return value. The memory provided to the agent
         *              function for both retval and varg is not preserved
         *              after this function call! If the memory is to be kept
         *              by the agent, use the manually marshaled version
         *              of this function. Also note that even though structures
         *              can be marshaled automatically, in this function, the
         *              type of the structure is unknown and therefore it must
         *              be handled manually, even though the marshaling is
         *              transparent to the user.
         * 
         * \sa          Overloaded CallAgentFunc, LibMCConsole example
         */
        public int CallAgentFunc(String funcName, ref object retval, ref object varg)
        {
            int temp = -1;
            IntPtr vargp = IntPtr.Zero, retvalp = IntPtr.Zero;
            try
            {
                vargp = Marshal.AllocHGlobal(Marshal.SizeOf(varg));
                Marshal.StructureToPtr(varg, vargp, false);
                retvalp = Marshal.AllocHGlobal(Marshal.SizeOf(retval));

                temp = CallAgentFunc(funcName, // The function to call
                    retvalp, // A place for the return value (void*)
                    vargp);  // The marshaled pointer to the argument (a structure)
                retval = Marshal.PtrToStructure(retvalp, retval.GetType());
                varg = Marshal.PtrToStructure(vargp, varg.GetType());
            }
            catch (OutOfMemoryException ex)
            {
                Console.WriteLine("Error allocating memory for calling agent functions: " + ex.Message);
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine("Error marshaling arguments for calling agent functions: " + ex.Message);
            }
            finally
            {
                Marshal.FreeHGlobal(vargp);
                Marshal.FreeHGlobal(retvalp);
            }

            return temp;
        }

        /**
         * \brief       Gets an agent's Ch interpreter.
         * 
         * Gets a pointer to the agent's Ch interpreter. Will be improved shortly.
         * 
         * \returns     A pointer to the Ch interpreter.
         * \note        Nothing in the LibMC.NET library can make use of the Ch 
         *              interpreter yet.
         * 
         * \todo        Wrap MC_GetAgentExecEngine with an object for the void* 
         *              pointer return type (Ch interpreter).
         */
        public IntPtr GetAgentExecEngine()
        {
            return MCAgency._MC_GetAgentExecEngine(Agent);
        }

        /**
         * \brief       Calls a function in an agent script.
         * 
         * Calls a function in an agent's script file. This function
         * requires manual marshaling by the user.
         * 
         * \param       task_num Task number to get data from
         * \param       data A pointer to memory for the data
         * \param       dim A pointer to hold the dimensions of the data
         * \param       extent A pointer to hold the dimensions of the data
         * \returns     The return value of the underlying MC_GetAgentReturnData
         *              function call.
         * 
         * \note        This function does nothing but throw an exception right now.
         * \todo        Implement GetAgentReturnData
         */
        public int GetAgentReturnData(int task_num, IntPtr data, IntPtr dim, IntPtr extent)
        {
            //return _MC_GetAgentReturnData(IntPtr agent, int task_num, IntPtr data, IntPtr dim, IntPtr extent);
            throw new Exception("GetAgentReturnData is not yet implemented!");
        }
    }

    /**
     * \brief   Exception class for use with null agent pointers
     * 
     * This class provides a way to inform the program that an
     * agent was created or accessed that had an invalid
     * internal pointer.
     */
    public class InvalidAgentException : SystemException
    {
        private const String msg = "Private agent pointer not valid.";

        /**
         * \brief   Null agent pointer exception constructor.
         * 
         * Constructor for the exception class. This exception
         * simply defines a recognizable exception class and sets
         * the exception message appropriately.
         */
        public InvalidAgentException()
            : base(msg)
        {
        }

        /**
         * \brief   Null agent pointer exception constructor.
         * 
         * Constructor for the exception class. Allows the use
         * of a custom error message.
         */
        public InvalidAgentException(String exc)
            : base(exc)
        {
        }
    }
}
