using System;
using System.Collections.Generic;
using System.Text;

/**
 * \page mcaclmessage MCAclMessage
 * Examples of commonly used MCAclMessage operations:
 * 
 * Create a new, blank ACL message:
 * \code
 * MCAclMessage tmp = new MCAclMessage();
 * tmp.New();
 * \endcode
 * 
 * Set the performative field:
 * \code
 * tmp.SetPerformative(MCAclMessage.MC_FipaPerformative_e.FIPA_INFORM);
 * \endcode
 * 
 * Set the sender:
 * \code
 * tmp.SetSender("agency", "http://" + host + ":" + 
 *     localport.ToString() + "/acc");
 * \endcode
 * 
 * Add an alternate reply-to field:
 * \code
 * tmp.AddReplyTo("mobagent2", "http://" + host + ":" + 
 *     localport.ToString() + "/acc");
 * \endcode
 * 
 * Add a receiver to the message:
 * \code
 * tmp.AddReceiver("mobagent1", "http://" + host + ":" + 
 *     localport.ToString() + "/acc");
 * \endcode
 * 
 * Set the content of the message:
 * \code
 * tmp.SetContent("This is content. Yay!");
 * \endcode
 * 
 * Finally, send and destroy the message:
 * \code
 * Agency.AclSend(tmp);
 * tmp.Destroy();
 * \endcode
 * 
 * Note that messages contain a pointer to allocated unmanaged memory 
 * and need to be disposed of after they are used. The agency creates
 * a copy of the message when it is sent, and therefore the MCAclMessage 
 * object is no longer needed.
 */

/** \file MCAclMessage.cs
 * Defines the MCAclMessage object and its member functions.
 */

/** \example LibMCFipaTest/Program.cs
 * Mobile-C FIPA ACL message demo program.
 */

namespace LibMC
{
    /**
     * \brief       Encapsulates ACL messages in the Mobile-C library.
     * 
     * This class contains a pointer to an ACL message in the Mobile-C library.
     * Functions are provided to send the message, set its various fields, and
     * destory the message.
     */
    public class MCAclMessage
    {
        /**
         * \brief   Enum for describing the type of an ACL message.
         * 
         * \note    This enum is pulled directly from the Mobile-C library.
         */
        public enum MC_FipaPerformative_e
        {
            FIPA_ERROR = -1,            /*!< Fipa performative enum value */
            FIPA_ZERO,                  /*!< Fipa performative enum value */
            FIPA_ACCEPT_PROPOSAL,       /*!< Fipa performative enum value */
            FIPA_AGREE,                 /*!< Fipa performative enum value */
            FIPA_CANCEL,                /*!< Fipa performative enum value */
            FIPA_CALL_FOR_PROPOSAL,     /*!< Fipa performative enum value */
            FIPA_CONFIRM,               /*!< Fipa performative enum value */
            FIPA_DISCONFIRM,            /*!< Fipa performative enum value */
            FIPA_FAILURE,               /*!< Fipa performative enum value */
            FIPA_INFORM,                /*!< Fipa performative enum value */
            FIPA_INFORM_IF,             /*!< Fipa performative enum value */
            FIPA_INFORM_REF,            /*!< Fipa performative enum value */
            FIPA_NOT_UNDERSTOOD,        /*!< Fipa performative enum value */
            FIPA_PROPOGATE,             /*!< Fipa performative enum value */
            FIPA_PROPOSE,               /*!< Fipa performative enum value */
            FIPA_PROXY,                 /*!< Fipa performative enum value */
            FIPA_QUERY_IF,              /*!< Fipa performative enum value */
            FIPA_QUERY_REF,             /*!< Fipa performative enum value */
            FIPA_REFUSE,                /*!< Fipa performative enum value */
            FIPA_REJECT_PROPOSAL,       /*!< Fipa performative enum value */
            FIPA_REQUEST,               /*!< Fipa performative enum value */
            FIPA_REQUEST_WHEN,          /*!< Fipa performative enum value */
            FIPA_REQUEST_WHENEVER,      /*!< Fipa performative enum value */
            FIPA_SUBSCRIBE              /*!< Fipa performative enum value */
        };

        private IntPtr aclmsg_p;

        /**
         * \brief       Default constructor.
         * 
         * Creates an empty ACL message object.
         */
        public MCAclMessage()
        {
            // NOT using accessor!
            aclmsg_p = IntPtr.Zero;
        }

        internal MCAclMessage(IntPtr msg)
        {
            AclMsg = msg;
        }
        
        /*
         * Accessors
         */
        internal IntPtr AclMsg
        {
            get
            {
                if (aclmsg_p == IntPtr.Zero)
                    throw new SystemException("Private acl message pointer is zero!");
                else
                    return aclmsg_p;
            }
            set
            {
                if (aclmsg_p == IntPtr.Zero)
                {
                    aclmsg_p = value;
                }
                else
                    throw new SystemException("Attempting to assign new acl message to non-zero acl message pointer!");
            }
        }

        /**
         * \brief       Creates a new, blank ACL message.
         * 
         * Creates a new ACL message. The message is blank but valid.
         */
        public void New()
        {
            AclMsg = MCAgency._MC_AclNew();
        }

        /**
         * \brief       Creates an ACL message that is a response to the argument.
         * 
         * Creates an ACL message to respond to the argument.
         * 
         * \param       acl_message The message from which to create the reply.
         * \returns     A new ACL message that is a response to the argument or
         *              an empty message if there is an error.
         */
        public MCAclMessage Reply(MCAclMessage acl_message)
        {
            IntPtr temp = MCAgency._MC_AclReply(acl_message.AclMsg);
            if (temp == IntPtr.Zero)
                return new MCAclMessage();
            else
                return new MCAclMessage(temp);
        }

        /**
         * \brief       Sets the performative field of the message.
         * 
         * Sets the performative field of the message.
         * 
         * \param       performative The fipa_performative_e enum describing the message.
         * \returns     The return value of the underlying MC_AclSetPerformative function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetPerformative(MC_FipaPerformative_e performative)
        {
            return MCAgency._MC_AclSetPerformative(AclMsg, performative);
        }

        /**
         * \brief       Sets the sender field of the message.
         * 
         * Sets the performative field of the message.
         * 
         * \param       name The name of the sending entity.
         * \param       address The address of the sending entity.
         * \returns     The return value of the underlying MC_AclSetSender function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetSender(String name, String address)
        {
            return MCAgency._MC_AclSetSender(AclMsg, name, address);
        }

        /**
         * \brief       Adds a receiver to the list of receivers.
         * 
         * Adds a receiver to the list of receivers for the message.
         * 
         * \param       name The name of the receiver.
         * \param       address The address of the receiver.
         * \returns     The return value of the underlying MC_AclAddReceiver function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int AddReceiver(String name, String address)
        {
            return MCAgency._MC_AclAddReceiver(AclMsg, name, address);
        }

        /**
         * \brief       Adds a "reply-to" field to the message.
         * 
         * Adds a "reply-to" field to the message. The reply-to field overrides
         * the sender field when creating a reply.
         * 
         * \param       name The name of the receiver.
         * \param       address The address of the receiver.
         * \returns     The return value of the underlying MC_AclAddAddReplyTo function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int AddReplyTo(String name, String address)
        {
            return MCAgency._MC_AclAddReplyTo(AclMsg, name, address);
        }

        /**
         * \brief       Sets the content field of the message.
         * 
         * Sets the content field of the message.
         * 
         * \param       content The string to copy to the content field.
         * \returns     The return value of the underlying MC_AclSetContent function.
         * 
         * \note        The message must be a valid message or this function will fail.
         */
        public int SetContent(String content)
        {
            return MCAgency._MC_AclSetContent(AclMsg, content);
        }

        /**
         * \brief       Destroys a message.
         * 
         * This function destroys a message in the Mobile-C library.
         * It releases the underlying memory and must be called when the message
         * is no longer needed. 
         * 
         * \returns     The return value of the underlying MC_AclDestroy function.
         * 
         * \note        The message must be a valid message or this function will fail.
         *              In addition, messages are not automatically destroyed by the 
         *              garbage collector. Use care when creating messages and ensure
         *              they are properly destroyed.
         */
        public int Destroy()
        {
            int temp = MCAgency._MC_AclDestroy(AclMsg);
            if (temp == 0)
                aclmsg_p = IntPtr.Zero;
            return temp;
        }

    }
}
