/*[
 * Copyright (c) 2007 Integration Engineering Laboratory
                      University of California, Davis
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the complete modified source code.  Modifications are to
 * be distributed as patches to the released version.  Permission to
 * distribute binaries produced by compiling modified sources is granted,
 * provided you
 *   1. distribute the corresponding source modifications from the
 *    released version in the form of a patch file along with the binaries,
 *   2. add special version identification to distinguish your version
 *    in addition to the base release version number,
 *   3. provide your name and address as the primary contact for the
 *    support of your modified version, and
 *   4. retain our contact information in regard to use of the base
 *    software.
 * Permission to distribute the released version of the source code along
 * with corresponding source modifications in the form of a patch file is
 * granted with same provisions 2 through 4 for binary distributions.
 *
 * This software is provided "as is" without express or implied warranty
 * to the extent permitted by applicable law.
]*/

#ifndef _WIN32
#include <unistd.h>
#include <pthread.h>
#else
#include <windows.h>
#endif
#include <embedch.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32
#include <sys/time.h>
#else
#include <time.h>
#endif

#include "include/libmc.h"
#include "include/macros.h"
#include "include/mc_platform.h"
#include "include/message.h"
#include "include/data_structures.h"
#include "include/fipa_acl_envelope.h"
#include "include/fipa_acl.h"
#include "include/agent.h"
#include "include/agent_task.h"
#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX 255
#endif

/* The Global platform variable */
mc_platform_p g_mc_platform;

/* **************************** *
 * Libmc Binary Space Functions *
 * **************************** */

int 
MC_AclDestroy(struct fipa_acl_message_s* message)
{
  return fipa_acl_message_Destroy(message);
}

EXPORTMC fipa_acl_message_t* 
MC_AclNew(void) {
  return fipa_acl_message_New();
}

EXPORTMC int 
MC_AclPost(MCAgent_t agent, struct fipa_acl_message_s* message)
{
  return agent_mailbox_Post(agent->mailbox, message);
}

EXPORTMC fipa_acl_message_t*
MC_AclReply(fipa_acl_message_t* acl_message)
{
  return fipa_Reply(acl_message);
}

EXPORTMC fipa_acl_message_t*
MC_AclRetrieve(MCAgent_t agent)
{
  return agent_mailbox_Retrieve(agent->mailbox);
}

EXPORTMC int
MC_AclSend(MCAgency_t attr, fipa_acl_message_t* acl)
{
  /* FIXME: We may want to multithread this later for asynchronous
   * sending to multiple hosts. But for now, we will use a simple
   * loop. */
  int i;
  int err;
  mtp_http_t* msg;
  dynstring_t* msg_string;
  message_p mc_message;

  char* host;
  int port;
  char* target;
  MCAgent_t agent;
  int num_addresses = 0;

  err = fipa_acl_Compose(&msg_string, acl);
  if( err ) {
    fprintf(stderr, "ACL Message Compose Error. %s:%d\n", __FILE__, __LINE__);
    return err;
  }
  for(i = 0; i < acl->receiver->num; i++) {
    /* The receiver may or may not have an address. If it does, send the
     * message to that address. If not, assume that the target agent is
     * local and try to send to that agent. */
    if (acl->receiver->fipa_agent_identifiers[i]->addresses == NULL) {
      num_addresses = 0;
    } else {
      num_addresses = acl->receiver->fipa_agent_identifiers[i]->addresses->num;
    }
    if (num_addresses == 0) {
      agent = MC_FindAgentByName(
          attr,
          acl->receiver->fipa_agent_identifiers[i]->name );
      if (agent == NULL) {
        fprintf(stderr, "Could not find local agent:%s. %s:%d\n",
            acl->receiver->fipa_agent_identifiers[i]->name,
            __FILE__, __LINE__);
      }
      MC_AclPost(agent, acl);
    } else {
      msg = mtp_http_New();
      /* Send to the first address listed */
      err = http_to_hostport(
          acl->receiver->fipa_agent_identifiers[i]->addresses->urls[0]->str,
          &host,
          &port,
          &target );
      if (err) {
        fprintf(stderr, "Invalid address. %s:%d\n", __FILE__, __LINE__);
        return err;
      }
      msg->host = strdup(host);
      msg->target = strdup(target);
      msg->message_parts = 2;
      msg->content = (mtp_http_content_t *)malloc(
          sizeof(mtp_http_content_t) * 2);

      /* Set up the message envelope */
      msg->content[0].data = (void*)fipa_envelope_Compose(acl);
      
      msg->content[0].content_type = strdup("application/xml");

      /* Set up the ACL message */
      msg->content[1].data = (void*)strdup(msg_string->message);
      msg->content[1].content_type = strdup("application/text");

      mc_message = mtp_http_CreateMessage(
          msg,
          host,
          port );
      mc_message->message_type = FIPA_ACL;
      mc_message->target = strdup("acc");

      message_Send
        (
         mc_message
        );
      message_Destroy(mc_message);
      mtp_http_Destroy(msg);
      free(host);
      free(target);
    }
  }
  dynstring_Destroy(msg_string);
  return 0;
}

EXPORTMC fipa_acl_message_t* 
MC_AclWaitRetrieve(MCAgent_t agent)
{
  return agent_mailbox_WaitRetrieve(agent->mailbox);
}

/* ACL Helper Functions Here */

int MC_AclSetPerformative(
    fipa_acl_message_t* acl,
    enum fipa_performative_e performative )
{
  acl->performative = performative;
  return 0;
}

int MC_AclSetSender(
    fipa_acl_message_t* acl,
    const char* name,
    const char* address )
{
  if(acl->sender != NULL) {
    /* There is already a sender. Lets over-write it */
    fipa_agent_identifier_Destroy(acl->sender);
  }
  acl->sender = fipa_agent_identifier_New();
  acl->sender->name = strdup(name);
  if (address != NULL) {
    acl->sender->addresses = fipa_url_sequence_New();
    acl->sender->addresses->num = 1;
    acl->sender->addresses->urls = (struct fipa_url_s**)malloc(
        sizeof(struct fipa_url_s*));
    acl->sender->addresses->urls[0] = fipa_url_New();
    acl->sender->addresses->urls[0]->str = strdup(address);
  }

  return 0;
}
  
int MC_AclAddReceiver(
    fipa_acl_message_t* acl,
    const char* name,
    const char* address )
{
  int i;
  struct fipa_agent_identifier_s** tmp;
  if (acl->receiver == NULL) {
    acl->receiver = fipa_agent_identifier_set_New();
  }
  acl->receiver_num++;

  acl->receiver->num++;
  tmp = (struct fipa_agent_identifier_s**)malloc(
      sizeof(struct fipa_agent_identifier_s*)
      * acl->receiver->num);
  /* Copy existing addresses to new array */
  for(i = 0; i < acl->receiver->num-1; i++) {
    tmp[i] = acl->receiver->fipa_agent_identifiers[i];
  }
  /* Create new receiver */
  tmp[i] = fipa_agent_identifier_New();
  tmp[i]->name = strdup(name);
  if(address != NULL) {
    tmp[i]->addresses = fipa_url_sequence_New();
    tmp[i]->addresses->num = 1;
    tmp[i]->addresses->urls = (struct fipa_url_s**)malloc(
        sizeof(struct fipa_url_s*));
    tmp[i]->addresses->urls[0] = fipa_url_New();
    tmp[i]->addresses->urls[0]->str = strdup(address);
  }
  free(acl->receiver->fipa_agent_identifiers);
  acl->receiver->fipa_agent_identifiers = tmp;
  return 0;
}

int MC_AclAddReplyTo(
    fipa_acl_message_t* acl,
    const char* name,
    const char* address)
{
  int i;
  struct fipa_agent_identifier_s** tmp;
  if (acl->reply_to == NULL) {
    acl->reply_to = fipa_agent_identifier_set_New();
  }

  acl->reply_to->num++;
  tmp = (struct fipa_agent_identifier_s**)malloc(
      sizeof(struct fipa_agent_identifier_s*)
      * acl->reply_to->num);
  /* Copy existing addresses to new array */
  for(i = 0; i < acl->reply_to->num-1; i++) {
    tmp[i] = acl->reply_to->fipa_agent_identifiers[i];
  }
  /* Create new reply_to */
  tmp[i] = fipa_agent_identifier_New();
  tmp[i]->name = strdup(name);
  if(address != NULL) {
    tmp[i]->addresses = fipa_url_sequence_New();
    tmp[i]->addresses->num = 1;
    tmp[i]->addresses->urls = (struct fipa_url_s**)malloc(
        sizeof(struct fipa_url_s*));
    tmp[i]->addresses->urls[0] = fipa_url_New();
    tmp[i]->addresses->urls[0]->str = strdup(address);
  }
  free (acl->reply_to->fipa_agent_identifiers);
  acl->reply_to->fipa_agent_identifiers = tmp;
  return 0;
}

int MC_AclSetContent(
    fipa_acl_message_t* acl,
    const char* content )
{
  if (acl->content != NULL) {
    /* There is already content. Lets over-write it. */
    fipa_string_Destroy(acl->content);
  }
  acl->content = fipa_string_New();
  acl->content->content = strdup(content);

  return 0;
}

/* End ACL Helper Functions */

EXPORTMC int 
MC_AddAgent(MCAgency_t attr, MCAgent_t agent) /*{{{*/
{
  agent->mc_platform = attr->mc_platform;

  agent_queue_Add(attr->mc_platform->agent_queue, agent);

  MUTEX_LOCK(attr->mc_platform->ams->runflag_lock);
  attr->mc_platform->ams->run = 1;
  COND_SIGNAL(attr->mc_platform->ams->runflag_cond);
  MUTEX_UNLOCK(attr->mc_platform->ams->runflag_lock);
  return 0;
} /*}}}*/

const void* MC_AgentVariableRetrieve(MCAgent_t agent, const char* var_name, int task_num)
{
  interpreter_variable_data_t* interp_var;

  if (task_num >= agent->datastate->task_progress) {
    return NULL;
  }

  interp_var = agent_variable_list_Search(
      agent->datastate->tasks[task_num]->agent_variable_list,
      var_name );
  if (interp_var == NULL) {
    return NULL;
  }

  return interp_var->data;
}

int MC_AgentVariableSave(MCAgent_t agent, const char* var_name)
{
  int current_task = agent->datastate->task_progress;
  const int default_num_vars = 50;
  agent_task_p task = agent->datastate->tasks[current_task];

  if(task->num_saved_variables == 0) {
    task->saved_variables = (char**)malloc(sizeof(char*)*default_num_vars);
    memset(task->saved_variables, 0, sizeof(char*)*default_num_vars);
  }
  task->saved_variables[task->num_saved_variables] = strdup(var_name);
  if(task->saved_variables[task->num_saved_variables] == NULL) {
    fprintf(stderr, "Memory error. %s:%d\n", __FILE__, __LINE__);
    return MC_ERR_MEMORY;
  }
  task->num_saved_variables++;

  return 0;
}

int 
MC_Barrier(MCAgency_t attr, int id) /*{{{*/
{
    barrier_queue_p list = attr->mc_platform->barrier_queue;
    barrier_node_p node;
    node = barrier_queue_Get(list, id);
    if(node == NULL) {
        return MC_ERR_NOT_FOUND;
    }

    MUTEX_LOCK(node->lock);
    node->num_waiting++;
    if (node->num_waiting >= node->num_registered) {
        /* Wake all agents/threads waiting on this barrier */
        COND_BROADCAST(node->cond);
        MUTEX_UNLOCK(node->lock);
        return MC_SUCCESS;
    } else {
        while (node->num_waiting < node->num_registered) {
            COND_WAIT(node->cond, node->lock);
        }
        MUTEX_UNLOCK(node->lock);
    }
    return MC_SUCCESS;
} /*}}}*/

EXPORTMC int
MC_BarrierInit(MCAgency_t attr, int id, int num_procs) /*{{{*/
{
    barrier_node_p node;
    /* First see if there already exists a barrier of the same ID. */
    node = barrier_queue_Get(attr->mc_platform->barrier_queue, id);
    if (node != NULL) {
      return MC_ERR;
    }
    node = barrier_node_Initialize(id, num_procs);
    barrier_queue_Add(attr->mc_platform->barrier_queue, node);
    return MC_SUCCESS;
} /*}}}*/

EXPORTMC int 
MC_BarrierDelete(MCAgency_t attr, int id) /*{{{*/
{
  return barrier_queue_Delete(id, attr->mc_platform->barrier_queue);
} /*}}}*/

EXPORTMC int 
MC_CallAgentFunc( /*{{{*/
        MCAgent_t agent,
        const char* funcName,
        void* returnVal, /* FIXME: Should this be a double pointer? */
        void* varg)
{
    int return_code;
    MUTEX_LOCK(agent->run_lock); 
    return_code = Ch_CallFuncByName(
            agent->agent_interp,
            funcName,
            returnVal,
            varg);
    MUTEX_UNLOCK(agent->run_lock); 
    return return_code;
} /*}}}*/ 

EXPORTMC int 
MC_ChInitializeOptions(MCAgency_t attr, ChOptions_t *options) { /*{{{*/
    if(attr->mc_platform == NULL) {
        fprintf(stderr, "MC_ChInitializeOptions must be called after MC_Start()\n");
        fprintf(stderr, "Using default interpretor options...\n");
        return 1;
    }
    else {
        if (attr->mc_platform->interp_options == NULL) {
          attr->mc_platform->interp_options = (ChOptions_t*)malloc(
              sizeof(ChOptions_t) );
        }
        *attr->mc_platform->interp_options = *options;
        attr->mc_platform->interp_options->chhome = strdup(options->chhome);
        return 0;
    }
} /*}}}*/

MCAgent_t
MC_ComposeAgent(
    const char* name,
    const char* home,
    const char* owner, 
    const char* code,
    const char* return_var_name,
    const char* server,
    int persistent
    )
{
  agent_p agent;
  agent = agent_New();
  if (agent == NULL) return NULL;
  agent->name = strdup(name);
  agent->home = strdup(home);
  agent->owner = strdup(owner);

  agent->orphan = 1;

  agent->agent_type = MC_LOCAL_AGENT;
  agent->agent_status = MC_WAIT_MESSGSEND;

  agent->datastate = agent_datastate_New();
  agent->datastate->number_of_tasks = 1;
  agent->datastate->persistent = persistent;
  agent->datastate->agent_code_ids = (char**)malloc(
      sizeof(char*)*2);
  if(agent->datastate->agent_code_ids == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }
  agent->datastate->agent_code_ids[0] = strdup("");
  if(agent->datastate->agent_code_ids[0] == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }
  agent->datastate->agent_code_ids[1] = NULL;

  agent->datastate->agent_codes = (char**)malloc(
      sizeof(char*)*2);
  if(agent->datastate->agent_codes == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }
  agent->datastate->agent_codes[0] = strdup(code);
  if(agent->datastate->agent_codes[0] == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }
  agent->datastate->agent_codes[1] = NULL;

  agent->datastate->agent_code = agent->datastate->agent_codes[0];

  agent->datastate->tasks = (agent_task_t**)malloc(
      sizeof(agent_task_t*));
  agent->datastate->tasks[0] = agent_task_New();
  if(return_var_name == NULL) {
    agent->datastate->tasks[0]->var_name = strdup("no-return");
  } else {
    agent->datastate->tasks[0]->var_name = strdup(return_var_name);
  }
  if(agent->datastate->tasks[0]->var_name == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }

  agent->datastate->tasks[0]->server_name = strdup(server);
  if(agent->datastate->tasks[0]->server_name == NULL) {
    fprintf(stderr, "Memory Error %s:%d\n", __FILE__, __LINE__);
  }
  
  return agent;
}

EXPORTMC int 
MC_CondBroadcast(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    condnode->signalled=1;
    COND_BROADCAST(condnode->cond);
    MUTEX_UNLOCK(condnode->lock);
    return 0;
} /*}}}*/

EXPORTMC int 
MC_CondSignal(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    condnode->signalled=1;
    COND_SIGNAL(condnode->cond);
    MUTEX_UNLOCK(condnode->lock);
    return 0;
} /*}}}*/

EXPORTMC int 
MC_CondWait(MCAgency_t attr, int id)  /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    if (condnode->signalled) {
        MUTEX_UNLOCK(condnode->lock);
        return 1;
    }

    while (condnode->signalled == 0) {
      COND_WAIT(condnode->cond, condnode->lock);
    }
    MUTEX_UNLOCK(condnode->lock);

    return 0;
} /*}}}*/

EXPORTMC int 
MC_CondReset(MCAgency_t attr, int id) /*{{{*/
{
    syncListNode_t *condnode;
    condnode = syncListFind(id, attr->mc_platform->syncList);
    if (condnode == NULL) {
        return MC_ERR_NOT_FOUND;
    }
    MUTEX_LOCK(condnode->lock);
    if (condnode->signalled) {
        condnode->signalled = 0;
        MUTEX_UNLOCK(condnode->lock);
        return 0;
    }
    MUTEX_UNLOCK(condnode->lock);
    return 1;
} /*}}}*/

int 
MC_CopyAgent(MCAgent_t* agent_out, const MCAgent_t agent_in) /*{{{*/
{
  *agent_out = agent_Copy(agent_in);
  return MC_SUCCESS;
} /*}}}*/

EXPORTMC int 
MC_DeleteAgent(MCAgent_t agent) /*{{{*/
{
    /* Error Checking */
    CHECK_NULL(agent, return MC_ERR_INVALID;);

    /* First, make sure the agent is no longer running */
    MC_TerminateAgent(agent);

    /* Now, we just set it's status to MC_WAIT_FINISHED so that the AMS will
     * flush it next chance it gets. */
    MC_SetAgentStatus(agent, MC_WAIT_FINISHED);
    return MC_SUCCESS;
} /*}}}*/

int MC_DestroyServiceSearchResult(
    char** agentName,
    char** serviceName,
    int* agentID,
    int numResult)
{
  int i;
  for(i = 0;i < numResult; i++)
  {
    free(agentName[i]);
    free(serviceName[i]);
  }
  free(agentName);
  free(serviceName);
  free(agentID);

  return 0;
}

int 
MC_DeregisterService( /*{{{*/
        MCAgency_t agency,
        int agentID,
        const char *serviceName)
{
    int err_code;

    df_request_list_node_t *req_node;
    df_deregister_p deregister_data;

    req_node = df_request_list_node_New();
    req_node->command = (char*)malloc(sizeof(char)*11);

    strcpy((char*)req_node->command, "deregister");

    deregister_data = (df_deregister_p)malloc(sizeof(df_deregister_t));
    deregister_data->agent_id = agentID;
    deregister_data->service_name = (char*)serviceName;

    req_node->data = deregister_data;

    err_code = df_AddRequest(
            agency->mc_platform->df,
            req_node
            );
    return err_code;
} /*}}}*/

EXPORTMC int
MC_End(MCAgency_t agency) /*{{{*/
{
  /* Now, we must stop all the running pthreads somehow... */
  /* We will set the quit flag and signal some important threads to make
   * sure they exit cleanly. We want the df, ams, and especially the acc
   * to be done with whatever they are doing before exiting. */
  MUTEX_LOCK(agency->mc_platform->quit_lock);
  agency->mc_platform->quit = 1;
  MUTEX_UNLOCK(agency->mc_platform->quit_lock);
  COND_SIGNAL(agency->mc_platform->df->request_list->cond);

  /* Stop the command prompt */
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_CP)) {
    THREAD_CANCEL( agency->mc_platform->cmd_prompt->thread );
  }

  /* Stop the listen thread */
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_ACC)) {
    THREAD_CANCEL( agency->mc_platform->acc->listen_thread );
  }

  /* Stop the connection queue */
  COND_SIGNAL(agency->mc_platform->connection_queue->cond);
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_ACC)) {
    THREAD_JOIN(agency->mc_platform->acc->thread);
  }

  /* Stop the message queue */
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_ACC)) {
    THREAD_JOIN(agency->mc_platform->acc->message_handler_thread);
  }

  /* Stop the AMS */
  COND_SIGNAL(agency->mc_platform->ams->runflag_cond);
  if( GET_THREAD_MODE( agency->threads, MC_THREAD_AMS)) {
    THREAD_JOIN(agency->mc_platform->ams->thread);
  }


  if( GET_THREAD_MODE( agency->threads, MC_THREAD_DF)) {
    THREAD_JOIN(agency->mc_platform->df->thread);
  }

  mc_platform_Destroy(agency->mc_platform);

  if (agency->hostName)
    free(agency->hostName); 
  free(agency);

  return 0;
} /*}}}*/

EXPORTMC MCAgent_t 
MC_FindAgentByName( MCAgency_t attr, /*{{{*/
    const char *name) 
{
  extern mc_platform_p g_mc_platform;
  if (attr == NULL) {
    return agent_queue_SearchName(g_mc_platform->agent_queue, name);
  } else {
    return agent_queue_SearchName(attr->mc_platform->agent_queue,
        name);
  }
} /*}}}*/

EXPORTMC MCAgent_t
MC_FindAgentByID( MCAgency_t attr, /*{{{*/
    int ID)
{
  extern mc_platform_p g_mc_platform;
  if (attr == NULL) {
    return agent_queue_Search(g_mc_platform->agent_queue, ID);
  } else {
    return agent_queue_Search(attr->mc_platform->agent_queue,
        ID);
  }
} /*}}}*/

#ifndef _WIN32
  time_t 
#else
  SYSTEMTIME
#endif
MC_GetAgentArrivalTime(MCAgent_t agent) /*{{{*/
{
  if (agent != NULL) {
    return agent->arrival_time;
  } else {
#ifndef _WIN32
    return (time_t)-1;
#else
    SYSTEMTIME oy;
    return oy;
#endif
  }
} /*}}}*/

EXPORTMC int 
MC_GetAgentStatus(MCAgent_t agent) /*{{{*/
{
  int status;
  MUTEX_LOCK(agent->lock);
  status = agent->agent_status;
  MUTEX_UNLOCK(agent->lock);
  return status;
} /*}}}*/

EXPORTMC char*
MC_GetAgentXMLString(MCAgent_t agent) /*{{{*/
{
  char *ret;
  ret = mxmlSaveAllocString(
      agent->datastate->xml_agent_root,
      NULL
      );
  return ret;
} /*}}}*/

/*ChInterp_t */
EXPORTMC void*
MC_GetAgentExecEngine(MCAgent_t agent) /*{{{*/
{
  return agent->agent_interp;
} /*}}}*/

EXPORTMC int 
MC_GetAgentID( /*{{{*/
    MCAgent_t agent
    )
{
  return agent->id;
} /*}}}*/

EXPORTMC char* 
MC_GetAgentName( /*{{{*/
    MCAgent_t agent
    )
{
  char *name;
  MUTEX_LOCK(agent->lock);
  name = (char*)malloc(sizeof(char) * 
      (strlen (agent->name) + 1)
      );
  strcpy(
      name,
      agent->name
      );
  MUTEX_UNLOCK(agent->lock);
  return name;
} /*}}}*/

EXPORTMC int
MC_GetAgentReturnData( /*{{{*/
    MCAgent_t agent,
    int task_num,
    void **data,
    int *dim,
    int **extent) 
{
  int num_elements;
  int size;
  int i;
  if (task_num >= agent->datastate->number_of_tasks) {
    *data = NULL;
    *dim = 0;
    *extent = NULL;
    return 1;
  }
  if (
      agent->datastate->tasks[task_num]->
      agent_return_data->data_type == -1
     )
  {
    return 1;
  }
  CH_DATATYPE_SIZE(
      agent->datastate->tasks[task_num]->agent_return_data->data_type,
      size);
  num_elements = 1;
  for (
      i = 0; 
      i < agent->datastate->tasks[task_num]->agent_return_data->array_dim;
      i++
      )
  {
    num_elements *= agent->datastate->
      tasks[task_num]->agent_return_data->array_extent[i];
  }


  *data = malloc(num_elements * size);
  memcpy(
      *data,
      agent->datastate->tasks[task_num]->
      agent_return_data->data,
      size * num_elements
      );
  *dim = agent->datastate->tasks[task_num]->agent_return_data->array_dim;
  *extent = (int*)malloc(
      sizeof(int) * 
      agent->datastate->tasks[task_num]->agent_return_data->array_dim
      );
  for (i = 0; i < *dim; i++) {
    (*extent)[i] = 
      agent->datastate->tasks[task_num]->agent_return_data->array_extent[i];
  }
  /*    memcpy(
   *extent,
   agent->datastate->tasks[0]->agent_return_data->array_extent,
   *dim
   ); */
  return 0;
} /*}}}*/

EXPORTMC int 
MC_GetAgentNumTasks(MCAgent_t agent) /*{{{*/
{
  return agent->datastate->number_of_tasks;
} /*}}}*/

EXPORTMC enum MC_AgentType_e
MC_GetAgentType(MCAgent_t agent) /*{{{*/
{
  if (agent != NULL) {
    return agent->agent_type;
  } else {
    return 0;
  }
} /*}}}*/

int 
MC_GetAllAgents(MCAgency_t attr, MCAgent_t **agents, int* num_agents) /*{{{*/
{
  int halt;
  int index = 0;
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  halt = (attr->mc_platform->giant == 1) ? 1 : 0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  if (halt)
    MC_HaltAgency(attr);
  /* Count the number of agents */
  while (agent_queue_SearchIndex(attr->mc_platform->agent_queue, index) != NULL) {
    index++;
  }
  *agents = (MCAgent_t *)malloc(sizeof(MCAgent_t*) * index);
  *num_agents = index;
  /* Assign the agents */
  index = 0;
  while
    (
     (
      (*agents)[index] = agent_queue_SearchIndex
      (
       attr->mc_platform->agent_queue,
       index
      )
     )
    )
    {
      index++;
    }
  if(halt)
    MC_ResumeAgency(attr);
  return 0;
} /*}}}*/

EXPORTMC int
MC_HaltAgency(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant=0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

EXPORTMC MCAgency_t 
MC_Initialize( /*{{{*/
    int port,
    MCAgencyOptions_t *options)
{
  MCAgency_t ret;
  int i=0;
  int options_malloc = 0;
  ret = (MCAgency_t)malloc(sizeof(struct agency_s));
  if (ret == NULL) {return NULL;}

  ret->hostName = malloc(HOST_NAME_MAX);
  if (ret->hostName == NULL) {return NULL;}
  gethostname(ret->hostName, HOST_NAME_MAX);
  /*Save some memory */
  CHECK_NULL(
      realloc(ret->hostName, sizeof(char)*(strlen(ret->hostName)+1)),
      return NULL;
      );
  if (ret->hostName == NULL) {return NULL;}
  ret->portno = port;
  ret->server = 1;
  ret->client = 0;
  ret->default_agentstatus = -1;
#ifdef MC_SECURITY
  ret->enable_security = 1;
#endif

  /* Set up agency options */
  if(options==NULL) {
    options = (MCAgencyOptions_t*)malloc(sizeof(MCAgencyOptions_t));
    MC_InitializeAgencyOptions(options);
    options_malloc = 1;
  }
  ret->threads = options->threads;
  ret->default_agentstatus = options->default_agent_status;
#ifdef MC_SECURITY
  ret->enable_security = options->enable_security;
#endif
  for(i = 0; i < MC_THREAD_ALL; i++) {
    ret->stack_size[i] = options->stack_size[i];
  }
  /* End agency options */

  ret->mc_platform = mc_platform_Initialize(ret);

  /* Set up the global platform */
  g_mc_platform = ret->mc_platform;

  if (options_malloc)
    free(options);

  return ret;
} /*}}}*/

EXPORTMC int 
MC_InitializeAgencyOptions(struct MCAgencyOptions_s* options) /*{{{*/
{
  int i;
  /* Set the default options */
  options->threads = 0xFFFF;
  options->default_agent_status = MC_WAIT_CH;
  options->modified = 0;
#ifdef MC_SECURITY
  options->enable_security = 1;
#endif
  for(i = 0; i < MC_THREAD_ALL; i++) {
    options->stack_size[i] = -1;
  }
  return 0;
} /*}}}*/

EXPORTMC int
MC_LoadAgentFromFile(MCAgency_t attr, const char* filename)
{
  struct stat filestat;
  char *buf;
  FILE *fp;
  message_p message;
  extern mc_platform_p g_mc_platform;
  buf = NULL;
  filestat.st_size = 0;
  stat(filename, &filestat);
  if (filestat.st_size != 0 ) {
    buf = malloc( sizeof(char) * (filestat.st_size+1) );
    memset(buf, 0, filestat.st_size+1);
  } else {
    fprintf(stderr, "Error: File %s not found.\n", filename);
    return 1;
  }

  fp = fopen(filename, "r");
  fread((void*)buf, filestat.st_size, 1, fp);
  fclose(fp);

  message = message_New();
  if ( 
      message_InitializeFromString (
      attr->mc_platform,
      message,
      buf,
      "",
      5050,
      "ams"
      )
     )
  {
    message_Destroy(message);
  }
  free(message->to_address);
  message->to_address = NULL;
  message->xml_root = mxmlLoadString(
      NULL,
      buf,
      NULL );
  if (message->xml_root == NULL) {
    fprintf(stderr, "Error loading agent. %s:%d\n", __FILE__, __LINE__);
    message_Destroy(message);
    return 1;
  }
  message->xml_payload = mxmlFindElement(
      message->xml_root,
      message->xml_root,
      "MOBILE_AGENT",
      NULL,
      NULL,
      MXML_DESCEND );
  if(message->xml_payload == NULL) {
    fprintf(stderr, "Error loading agent: <MOBILE_AGENT> tag not found. %s:%d\n",
        __FILE__, __LINE__ );
    message_Destroy(message);
    return 1;
  }

  message_queue_Add
    (
     attr->mc_platform->message_queue,
     message
    );
  return 0;
}

EXPORTMC int 
MC_MutexLock(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  MUTEX_LOCK(syncnode->lock);
  return 0;
} /*}}}*/

EXPORTMC int 
MC_MutexUnlock(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  MUTEX_UNLOCK(syncnode->lock);
  return 0;
} /*}}}*/

EXPORTMC int
MC_PrintAgentCode(MCAgent_t agent) /*{{{*/
{
  int progress;
  MUTEX_LOCK(agent->lock);
  progress = agent->datastate->task_progress;
  /* If progress is past the last task, print the last task's code. */
  if (progress >= agent->datastate->number_of_tasks) {
    progress = agent->datastate->number_of_tasks - 1;
  }
  printf("%s\n",
      agent->datastate->agent_code);
  MUTEX_UNLOCK(agent->lock);
  return 0;
} /*}}}*/

EXPORTMC int 
MC_RegisterService( /*{{{*/
    MCAgency_t agency,
    MCAgent_t agent,
    int agentID,
    const char *agentName,
    char **serviceNames,
    int numServices)
{
  df_request_list_node_t *req_node;
  df_node_t *new_node;
  int i;
  /* Error checking: Either an agent, or agent Name must be
   * provided. */
  if (agent == NULL && agentName == NULL) {
    return MC_ERR_INVALID_ARGS;
  }
  /* init the request node */
  req_node = df_request_list_node_New();
  req_node->command = (char*)malloc(sizeof(char)*9);
  strcpy((char*)req_node->command, "register");

  new_node = (df_node_t*)malloc(sizeof(df_node_t));
  CHECK_NULL(new_node, return MC_ERR_MEMORY);

  /* Init the lock */
  new_node->lock = (MUTEX_T*)malloc(sizeof(MUTEX_T));
  CHECK_NULL(new_node->lock, return MC_ERR_MEMORY);
  MUTEX_INIT(new_node->lock);

  /* Init the agentID */
  if (agent==NULL) {
    new_node->agent_id = agentID;
  } else {
    new_node->agent_id = agent->id;
  }

  /* Init the agent name */
  if (agent==NULL) {
    new_node->agent_name = 
      (char*)malloc(sizeof(char)*(strlen(agentName)+1));
    CHECK_NULL(new_node->agent_name, return MC_ERR_MEMORY;);
    strcpy(new_node->agent_name, agentName);
  } else {
    new_node->agent_name = 
      (char*)malloc(
          sizeof(char) * 
          (strlen(agent->name)+1)
          );
    CHECK_NULL(new_node->agent_name, return MC_ERR_MEMORY;);
    strcpy(new_node->agent_name, agent->name);
  }

  /* Init the services */
  new_node->service_names = (char**)malloc(
      sizeof(char*) * numServices
      );
  CHECK_NULL(new_node->service_names, return MC_ERR_MEMORY;);
  for (i = 0; i < numServices; i++) {
    new_node->service_names[i] = (char*) malloc(
        sizeof(char) * (strlen(serviceNames[i]) + 1)
        );
    CHECK_NULL(new_node->service_names[i], return MC_ERR_MEMORY;);
    strcpy(
        new_node->service_names[i],
        serviceNames[i]
        );
  }
  new_node->num_services = numServices;

  req_node->data = (void*)new_node;
  req_node->data_size = (sizeof(new_node));

  return df_AddRequest(
      agency->mc_platform->df,
      req_node
      );
} /*}}}*/

EXPORTMC int 
MC_ResumeAgency(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 1;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

EXPORTMC MCAgent_t
MC_RetrieveAgent(MCAgency_t attr) /*{{{*/
  /* This function retrieves the first agent with agent_status
     MC_AGENT_NEUTRAL it finds. If there are no agents with
     the specified attributes, return value is NULL. */
{
  int i;
  MCAgent_t agent=NULL, ret;
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  for (i = 0; i < attr->mc_platform->agent_queue->size; i++) {
    agent = ListSearch(
        attr->mc_platform->agent_queue->list, i);
    if (agent->agent_status == MC_AGENT_NEUTRAL) {
      break;
    }
  }
  if (agent == NULL) {
    MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
    return NULL;
  }
  if (agent->agent_status != MC_AGENT_NEUTRAL) {
    MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
    return NULL;
  }
  ret = (MCAgent_t)malloc(sizeof(agent_t));
  *ret = *agent;
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return ret;
}/*}}}*/

EXPORTMC char * 
MC_RetrieveAgentCode(MCAgent_t agent) /*{{{*/
{
  char *buf;
  int len, progress;
  MUTEX_LOCK(agent->lock);
  progress = agent->datastate->task_progress;
  len = strlen(
      agent->datastate->agent_code);
  buf = (char*)malloc( (len+1)*sizeof(char));
  strcpy(buf,
      agent->datastate->agent_code);
  MUTEX_UNLOCK(agent->lock);
  return buf;
} /*}}}*/

EXPORTMC int 
MC_ResetSignal(MCAgency_t attr) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 1;
  attr->mc_platform->MC_signal = MC_NO_SIGNAL;
  COND_SIGNAL(attr->mc_platform->giant_cond);
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

EXPORTMC int 
MC_SearchForService( /*{{{*/
    /* Input args */
    MCAgency_t attr, 
    const char *searchString,
    /* Return Args */
    char*** agentNames,
    char*** serviceNames,
    int** agentIDs,
    int* numResults)
{
  df_request_search_p search;
  df_search_results_p results;
  df_request_list_node_p request;
  search = df_request_search_New();
  CHECK_NULL(search, return MC_ERR_MEMORY;);
  results = (df_search_results_p)malloc(sizeof(df_search_results_t));
  CHECK_NULL(results, return MC_ERR_MEMORY;);
  request = df_request_list_node_New();
  CHECK_NULL(request, return MC_ERR_MEMORY;);


  search->search_results = results;
  search->search_string = (char*)searchString;

  request->data = (void*)search;
  request->command = malloc(sizeof(char) * 7);
  strcpy((char*)request->command, "search");
  request->data_size = sizeof(df_request_search_t);

  COND_SLEEP_ACTION(
      search->cond,
      search->lock,

      df_AddRequest(attr->mc_platform->df, request);
      );
  /* When we get here, search->results should be filled. */
  *agentNames = search->search_results->agent_names;
  *serviceNames = search->search_results->service_names;
  *agentIDs = search->search_results->agent_ids;
  *numResults = search->search_results->num_results;

  /* Free unused data structures */
  free((void*)request->command);
  df_request_list_node_Destroy(request);
  df_request_search_Destroy(search);

  return MC_SUCCESS;
} /*}}}*/

EXPORTMC int 
MC_SemaphorePost(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  SEMAPHORE_POST(syncnode->sem);
  return 0;
} /*}}}*/

EXPORTMC int 
MC_SemaphoreWait(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *syncnode;
  syncnode = syncListFind(id, attr->mc_platform->syncList);
  if (syncnode == NULL) {
    return 1;
  }
  SEMAPHORE_WAIT(syncnode->sem);
  return 0;
} /*}}}*/

int
MC_SendCh(MCAgency_t attr, /*{{{*/
    const char *filename,
    const char *remotehost,
    int port)
{
  printf("Sorry, not implemented yet.\n");
  return -1;
} /*}}}*/

EXPORTMC int 
MC_SendAgentMigrationMessage(MCAgency_t attr, /*{{{*/
    const char *string,
    const char *hostname,
    int port)
{
  message_p message;
  message = message_New();
  if(
      message_InitializeFromString
      (
       attr->mc_platform,
       message,
       string,
       hostname,
       port,
       "ams"
      )
    )
  {
    message_Destroy(message);
    return MC_ERR;
  } else {
    return message_queue_Add
      (
       attr->mc_platform->message_queue,
       message
      );
  }
} /*}}}*/

EXPORTMC int
MC_SendAgentMigrationMessageFile(MCAgency_t attr,  /*{{{*/
    const char *filename, 
    const char *hostname,
    int port)
{
  struct stat filestat;
  char *buf;
  FILE *fp;
  int ret;
  message_p message;
  extern mc_platform_p g_mc_platform;
  buf = NULL;
  filestat.st_size = 0;
  stat(filename, &filestat);
  if (filestat.st_size != 0 ) {
    buf = malloc( sizeof(char) * (filestat.st_size+1) );
    memset(buf, 0, filestat.st_size+1);
  } else {
    fprintf(stderr, "Error: File %s not found.\n", filename);
    return 1;
  }

  fp = fopen(filename, "r");
  fread((void*)buf, filestat.st_size, 1, fp);
  fclose(fp);

  if (attr!=NULL) {
    message = message_New();
    if( 
        message_InitializeFromString
        (
         attr->mc_platform,
         message,
         buf,
         hostname,
         port,
         "ams"
        )
      )
    {
      message_Destroy(message);
    } else {
      ret = message_queue_Add
        (
         attr->mc_platform->message_queue,
         message 
        );
    }
  } else {
    message = message_New();
    if(
        message_InitializeFromString
        (
         g_mc_platform,
         message,
         buf,
         hostname,
         port,
         "ams"
        )
      )
    {
      message_Destroy(message);
    } else {
      ret = message_queue_Add
        (
         g_mc_platform->message_queue,
         message
        );
    }
  }
  free(buf);
  return ret;
} /*}}}*/

EXPORTMC int 
MC_SendSteerCommand(MCAgency_t attr, enum MC_SteerCommand_e cmd) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->MC_steer_lock);
  attr->mc_platform->MC_steer_command = cmd;
  COND_BROADCAST(attr->mc_platform->MC_steer_cond);
  MUTEX_UNLOCK(attr->mc_platform->MC_steer_lock);
  return 0;
} /*}}}*/

int
MC_SetAgentStatus(MCAgent_t agent, int status) /*{{{*/
{
  MUTEX_LOCK(agent->lock);
  agent->agent_status = status;
  if (!agent->orphan) {
    MUTEX_LOCK(agent->mc_platform->ams->runflag_lock);
    agent->mc_platform->ams->run = 1;
    COND_SIGNAL(agent->mc_platform->ams->runflag_cond);
    MUTEX_UNLOCK(agent->mc_platform->ams->runflag_lock);
  }
  MUTEX_UNLOCK(agent->lock);
  return 0;
} /*}}}*/

int 
MC_SetDefaultAgentStatus(/*{{{*/
    MCAgency_t agency,
    enum MC_AgentStatus_e status
    ) 
{
  agency->mc_platform->default_agentstatus = status;
  return 0;
} /*}}}*/

EXPORTMC int 
MC_SetThreadOn(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index) /*{{{*/
{
  SET_THREAD_ON(options->threads, index);
  return 0;
} /*}}}*/

EXPORTMC int 
MC_SetThreadsAllOn(MCAgencyOptions_t* options)
{
  int i;
  for(i = 0; i < MC_THREAD_ALL; i++) {
    SET_THREAD_ON(options->threads, i);
  }
  return 0;
}

EXPORTMC int
MC_SetThreadOff(MCAgencyOptions_t *options, enum MC_ThreadIndex_e index) /*{{{*/
{
  SET_THREAD_OFF(options->threads, index);
  return 0;
} /*}}}*/

EXPORTMC int
MC_SetThreadsAllOff(MCAgencyOptions_t* options)
{
  int i;
  for(i = 0; i < MC_THREAD_ALL; i++) {
    SET_THREAD_OFF(options->threads, i);
  }
  return 0;
}

EXPORTMC int 
MC_Steer(                        /*{{{*/
    MCAgency_t attr,
    int (*funcptr)(void* data),
    void *arg
    )
{
  MUTEX_LOCK(attr->mc_platform->MC_steer_lock);
  do {
    attr->mc_platform->MC_steer_command = MC_RUN;
    MUTEX_UNLOCK(attr->mc_platform->MC_steer_lock);
    (*funcptr)(arg);
  } while 
  (
   attr->mc_platform->MC_steer_command == MC_RESTART 
  );
  return 0;
} /*}}}*/

EXPORTMC enum MC_SteerCommand_e 
MC_SteerControl(void) /*{{{*/
{
  extern mc_platform_p g_mc_platform;
  /* Check to see what current command is */
  MUTEX_LOCK(g_mc_platform->MC_steer_lock);
  while (g_mc_platform->MC_steer_command == MC_SUSPEND) {
    COND_WAIT(
        g_mc_platform->MC_steer_cond,
        g_mc_platform->MC_steer_lock
        );
  }
  MUTEX_UNLOCK(g_mc_platform->MC_steer_lock);
  return g_mc_platform->MC_steer_command;
} /*}}}*/

EXPORTMC int 
MC_SyncDelete(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *sync_node;
  /* First, lock the entire list. */
  MUTEX_LOCK(attr->mc_platform->syncList->giant_lock);

  /* Find and lock the node */
  sync_node = syncListFind(id, attr->mc_platform->syncList);
  if (sync_node == NULL) {
    MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);
    return MC_ERR_NOT_FOUND;
  }
  MUTEX_LOCK(sync_node->lock);

  /* Remove it from the list so it may no longer be used */
  if (syncListRemove(id, attr->mc_platform->syncList) == NULL) {
    fprintf(stderr, "Fatal error. %s:%d\n",
        __FILE__,
        __LINE__ );
    exit(0);
  }

  /* Now, unlock and destroy */
  MUTEX_UNLOCK(sync_node->lock);
  MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);

  return syncListNodeDestroy(sync_node);
} /*}}}*/

EXPORTMC int
MC_SyncInit(MCAgency_t attr, int id) /*{{{*/
{
  syncListNode_t *node;
  node = syncListNodeNew();
  MUTEX_LOCK(attr->mc_platform->syncList->giant_lock);
  if (id == 0) {
    id = rand();
  }
  while (
      syncListFind(id, attr->mc_platform->syncList) != NULL
      ) 
  {
    id = rand();
  }

  node->id = id;
  syncListAddNode(
      node,
      attr->mc_platform->syncList
      );
  MUTEX_UNLOCK(attr->mc_platform->syncList->giant_lock);
  return id;
}/*}}}*/

EXPORTMC int 
MC_TerminateAgent(MCAgent_t agent) /*{{{*/
{
  int status=0;
  if(agent->agent_interp != NULL) {
    status = Ch_Abort (agent->agent_interp);
  }
  return status;
} /*}}}*/


#ifdef _WIN32
EXPORTMC BOOL
MC_MainLoop(MCAgency_t attr) /*{{{*/
{
  /*    return CloseHandle(attr->global->gaf_ap->thread_id[attr->global->mr_index]); */
  Sleep (INFINITE);
  return 0;
}
#else
int 
MC_MainLoop(MCAgency_t attr) 
{
  return pthread_join(attr->mc_platform->ams->thread, NULL);
} /*}}}*/
#endif

EXPORTMC int
MC_WaitAgent(MCAgency_t attr) /*{{{*/
{
  int size;
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  while(1) {
    size = attr->mc_platform->agent_queue->size;
    COND_WAIT(
        attr->mc_platform->agent_queue->cond,
        attr->mc_platform->agent_queue->lock
        );
    if (size < attr->mc_platform->agent_queue->size) {
      MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
      break;
    } 
  }
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return 0;
} /*}}}*/

EXPORTMC MCAgent_t
MC_WaitRetrieveAgent(MCAgency_t attr) /*{{{*/
{
  int index;
  MCAgent_t agent;
  MC_WaitSignal(attr, MC_RECV_AGENT);
  MUTEX_LOCK(attr->mc_platform->agent_queue->lock);
  index = attr->mc_platform->agent_queue->size-1;
  agent = ListSearch(
      attr->mc_platform->agent_queue->list, index);
  MUTEX_UNLOCK(attr->mc_platform->agent_queue->lock);
  return agent;
} /*}}}*/

/* MC_WaitSignal */
/* This function blocks until one of the signals in argument
 * 'signals' is signalled. 'signals' must be a binary | combination
 * of enum MC_Signal_e type. */
EXPORTMC int
MC_WaitSignal(MCAgency_t attr, int signals) /*{{{*/
{
  MUTEX_LOCK(attr->mc_platform->MC_signal_lock);
  while(! (signals & attr->mc_platform->MC_signal)) {
    COND_WAIT(
        attr->mc_platform->MC_signal_cond,
        attr->mc_platform->MC_signal_lock
        );
  }
  MUTEX_UNLOCK(attr->mc_platform->MC_signal_lock);
  MUTEX_LOCK(attr->mc_platform->giant_lock);
  attr->mc_platform->giant = 0;
  MUTEX_UNLOCK(attr->mc_platform->giant_lock);
  return 0;
} /*}}}*/

/* *********************** *
 * Ch Space chdl functions *
 * *********************** */

/* MC_AclDestroy */
int MC_AclDestroy_chdl(void* varg)
{
  int retval;
  fipa_acl_message_t* acl_message;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  acl_message = Ch_VaArg(interp, ap, fipa_acl_message_t*);
  retval = MC_AclDestroy(acl_message);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclNew */
void* MC_AclNew_chdl(void* varg)
{
  void* retval;
  retval = (void*)MC_AclNew();
  return retval;
}

/* MC_AclPost */
int MC_AclPost_chdl(void* varg)
{
  int retval;
  agent_p agent;
  fipa_acl_message_t* acl_message;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, agent_p);
  acl_message = Ch_VaArg(interp, ap, fipa_acl_message_t*);
  retval = MC_AclPost(agent, acl_message);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclReply */
EXPORTCH void*
MC_AclReply_chdl(void* varg)
{
  void* retval;
  fipa_acl_message_t* acl_message;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  acl_message = Ch_VaArg(interp, ap, fipa_acl_message_t*);
  retval = (void*)MC_AclReply(acl_message);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclRetrieve */
EXPORTCH void*
MC_AclRetrieve_chdl(void* varg)
{
  void* retval;
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_AclRetrieve(agent);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclSend */
EXPORTCH int
MC_AclSend_chdl(void* varg)
{
  int retval;
  fipa_acl_message_t* acl_message;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  acl_message = (fipa_acl_message_t*) Ch_VaArg(interp, ap, void*);
  retval = MC_AclSend(temp_attr, acl_message);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
}

/* MC_AclWaitRetrieve */
EXPORTCH void*
MC_AclWaitRetrieve_chdl(void *varg)
{
  void* retval;
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_AclWaitRetrieve(agent);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* BEGIN Acl helper functions */

/* MC_AclSetPerformative */
EXPORTCH int
MC_AclSetPerformative_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct fipa_acl_message_s* acl;
  enum fipa_performative_e performative;
  int retval;

  Ch_VaStart(interp, ap, varg);
  acl = Ch_VaArg(interp, ap, struct fipa_acl_message_s*);
  performative = Ch_VaArg(interp, ap, enum fipa_performative_e);
  retval = MC_AclSetPerformative(acl, performative);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclSetSender */
EXPORTCH int
MC_AclSetSender_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct fipa_acl_message_s* acl;
  char* name;
  char* address;
  int retval;

  Ch_VaStart(interp, ap, varg);
  acl = Ch_VaArg(interp, ap, struct fipa_acl_message_s*);
  name = Ch_VaArg(interp, ap, char*);
  address = Ch_VaArg(interp, ap, char*);
  retval = MC_AclSetSender(acl, name, address);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclAddReceiver */
EXPORTCH int
MC_AclAddReceiver_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct fipa_acl_message_s* acl;
  char* name;
  char* address;
  int retval;

  Ch_VaStart(interp, ap, varg);
  acl = Ch_VaArg(interp, ap, struct fipa_acl_message_s*);
  name = Ch_VaArg(interp, ap, char*);
  address = Ch_VaArg(interp, ap, char*);
  retval = MC_AclAddReceiver(acl, name, address);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclAddReplyTo */
EXPORTCH int
MC_AclAddReplyTo_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct fipa_acl_message_s* acl;
  char* name;
  char* address;
  int retval;

  Ch_VaStart(interp, ap, varg);
  acl = Ch_VaArg(interp, ap, struct fipa_acl_message_s*);
  name = Ch_VaArg(interp, ap, char*);
  address = Ch_VaArg(interp, ap, char*);
  retval = MC_AclAddReplyTo(acl, name, address);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AclSetContent */
EXPORTCH int
MC_AclSetContent_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct fipa_acl_message_s* acl;
  char* content;
  int retval;

  Ch_VaStart(interp, ap, varg);
  acl = Ch_VaArg(interp, ap, struct fipa_acl_message_s*);
  content = Ch_VaArg(interp, ap, char*);
  retval = MC_AclSetContent(acl, content);
  Ch_VaEnd(interp, ap);
  return retval;
}

/* END Acl Helper Functions */

/* MC_AddAgent */
EXPORTCH int
MC_AddAgent_chdl(void *varg) /*{{{*/
{
  int retval;
  MCAgent_t agent;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agent = (MCAgent_t) Ch_VaArg(interp, ap, void*);
  retval = MC_AddAgent(temp_attr, agent);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_AgentVariableRetrieve */
EXPORTCH const void*
MC_AgentVariableRetrieve_chdl(void* varg)
{
  void* retval;
  MCAgent_t agent;
  const char* var_name;
  int task_num;

  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);

  agent = Ch_VaArg(interp, ap, MCAgent_t);
  var_name = Ch_VaArg(interp, ap, const char* );
  task_num = Ch_VaArg(interp, ap, int);

  retval = MC_AgentVariableRetrieve(agent, var_name, task_num);

  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_AgentVariableSave */
EXPORTCH int
MC_AgentVariableSave_chdl(void *varg)
{
  int retval;
  MCAgent_t agent;
  const char* var_name;

  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);

  agent = Ch_VaArg(interp, ap, MCAgent_t);
  var_name = Ch_VaArg(interp, ap, const char*);

  retval = MC_AgentVariableSave(agent, var_name);

  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_CallAgentFunc */
EXPORTCH int
MC_CallAgentFunc_chdl(void *varg) /*{{{*/
{
  int retval;
  /* Function Args */
  MCAgent_t agent;
  const char* funcName;
  void* returnVal;
  void* args;

  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);

  agent =     Ch_VaArg(interp, ap, MCAgent_t);
  funcName =  Ch_VaArg(interp, ap, const char*);
  returnVal = Ch_VaArg(interp, ap, void*);
  args =      Ch_VaArg(interp, ap, void*);

  retval = MC_CallAgentFunc(
      agent,
      funcName,
      returnVal,
      args);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_Barrier_chdl*/
EXPORTCH int
MC_Barrier_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_Barrier(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_BarrierDelete_chdl*/
EXPORTCH int
MC_BarrierDelete_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_BarrierDelete(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_BarrierInit_chdl*/
EXPORTCH int
MC_BarrierInit_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int num_procs;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  num_procs = Ch_VaArg(interp, ap, int);
  retval = MC_BarrierInit(temp_attr, id, num_procs);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondBroadcast_chdl*/
EXPORTCH int
MC_CondBroadcast_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondBroadcast(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_ComposeAgent */
EXPORTCH MCAgent_t 
MC_ComposeAgent_chdl(void *varg) /*{{{*/
{
  MCAgent_t retval;

  /* Function Args */
  const char* name;
  const char* home;
  const char* owner;  
  const char* code;
  const char* return_var_name;
  const char* server;
  int persistent;

  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);

  name  = Ch_VaArg(interp, ap, const char*);
  home  = Ch_VaArg(interp, ap, const char*);
  owner = Ch_VaArg(interp, ap, const char*);
  code = Ch_VaArg(interp, ap, const char*);
  return_var_name = Ch_VaArg(interp, ap, const char*);
  server = Ch_VaArg(interp, ap, const char*);
  persistent = Ch_VaArg(interp, ap, int);

  retval= MC_ComposeAgent(
      name,
      home,
      owner,
      code,
      return_var_name,
      server,
      persistent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_CondSignal_chdl*/
EXPORTCH int
MC_CondSignal_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondSignal(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondReset_chdl*/
EXPORTCH int
MC_CondReset_chdl(void *varg) /*{{{*/   
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondReset(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_CondWait_chdl*/
EXPORTCH int
MC_CondWait_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_CondWait(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

EXPORTCH int
MC_DeleteAgent_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_DeleteAgent(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

EXPORTCH int
MC_DestroyServiceSearchResult_chdl(void* varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  char** agentName;
  char** serviceName;
  int* agentID;
  int numResult;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agentName = Ch_VaArg(interp, ap, char**);
  serviceName = Ch_VaArg(interp, ap, char**);
  agentID = Ch_VaArg(interp, ap, int*);
  numResult = Ch_VaArg(interp, ap, int);

  retval = MC_DestroyServiceSearchResult(
      agentName,
      serviceName,
      agentID,
      numResult );
  Ch_VaEnd(interp, ap);
  return retval;
}

/* MC_DeregisterService_chdl */
EXPORTCH int
MC_DeregisterService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int agentID;
  char *serviceName;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agentID = Ch_VaArg(interp, ap, int);
  serviceName = (char*)Ch_VaArg(interp, ap, const char*);
  retval = MC_DeregisterService(
      temp_attr,
      agentID,
      serviceName );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

EXPORTCH int
MC_End_chdl(void *varg) /* {{{ */
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_End(temp_attr);

  return retval;
} /* }}} */

/* MC_FindAgentByID_chdl*/
EXPORTCH MCAgent_t
MC_FindAgentByID_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_FindAgentByID(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_FindAgentByName_chdl*/
EXPORTCH MCAgent_t
MC_FindAgentByName_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t retval;
  ChInterp_t interp;
  ChVaList_t ap;
  const char *name;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  name = Ch_VaArg(interp, ap, const char *);
  retval = MC_FindAgentByName(temp_attr, name);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_GetAgentArrivalTime */
#ifndef _WIN32
EXPORTCH time_t
#else
EXPORTCH SYSTEMTIME
#endif
MC_GetAgentArrivalTime_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
#ifndef _WIN32
  time_t arrival_time;
#else
  SYSTEMTIME arrival_time;
#endif

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  arrival_time = MC_GetAgentArrivalTime(agent);
  Ch_VaEnd(interp, ap);
  return arrival_time;
} /* }}} */

/* MC_GetAgentID */
EXPORTCH int
MC_GetAgentID_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  id = MC_GetAgentID(agent);
  Ch_VaEnd(interp, ap);
  return id;
} /*}}}*/

/* MC_GetAgentName */
EXPORTCH char*
MC_GetAgentName_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  char* name;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  name = MC_GetAgentName(agent);
  Ch_VaEnd(interp, ap);
  return name;
} /*}}}*/

/* MC_GetAgentNumTasks_chdl */
EXPORTCH int
MC_GetAgentNumTasks_chdl(void *varg)
{
  MCAgent_t agent;
  ChInterp_t interp;
  ChVaList_t ap;
  int num_tasks;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  num_tasks = MC_GetAgentNumTasks(agent);
  Ch_VaEnd(interp, ap);
  return num_tasks;
}

/* MC_GetAgentStatus_chdl */
EXPORTCH int
MC_GetAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  int status;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  status = MC_GetAgentStatus(agent);
  Ch_VaEnd(interp, ap);
  return status;
} /*}}}*/

/* MC_GetAgentXMLString_chdl */
EXPORTCH char *
MC_GetAgentXMLString_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  char *retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_GetAgentXMLString(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

#ifndef _WIN32
EXPORTCH int
MC_GetTimeOfDay_chdl(void *varg)
{
  ChInterp_t interp;
  ChVaList_t ap;
  struct timeval *tv;
  Ch_VaStart(interp, ap, varg);
  tv = Ch_VaArg(interp, ap, struct timeval*);
  gettimeofday(tv, NULL);
  Ch_VaEnd(interp, ap);
  return 0;
}
#endif

/* MC_HaltAgency_chdl */
EXPORTCH int
MC_HaltAgency_chdl(void *varg)
{
  MCAgency_t temp_attr;
  int retval;
  extern mc_platform_p g_mc_platform;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_HaltAgency(temp_attr);

  free(temp_attr);
  return retval;
}

/* MC_MutexLock_chdl */
EXPORTCH int
MC_MutexLock_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_MutexLock(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_MutexUnlock_chdl */
EXPORTCH int
MC_MutexUnlock_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_MutexUnlock(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_PrintAgentCode_chdl */
EXPORTCH int
MC_PrintAgentCode_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_PrintAgentCode(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_RegisterService_chdl */
EXPORTCH int
MC_RegisterService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  int retval;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  /* varg arguments */
  MCAgent_t agent;
  char **serviceNames;
  int numServices;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  serviceNames = Ch_VaArg(interp, ap, char **);
  numServices = Ch_VaArg(interp, ap, int);

  retval = MC_RegisterService(
      temp_attr,      /* agency */
      agent,          /* agent */ 
      0,              /* agent id */
      NULL,           /* agent name */
      serviceNames,   /* services */
      numServices
      );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_ResumeAgency_chdl */
EXPORTCH int
MC_ResumeAgency_chdl(void *varg)
{
  MCAgency_t temp_attr;
  int retval;
  extern mc_platform_p g_mc_platform;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  retval = MC_ResumeAgency(temp_attr);

  free(temp_attr);
  return retval;
}

/* MC_RetrieveAgent_chdl */
EXPORTCH MCAgent_t
MC_RetrieveAgent_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  MCAgent_t agent;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  agent = MC_RetrieveAgent(temp_attr);
  free(temp_attr);
  return agent;
} /*}}}*/

/* MC_RetrieveAgentCode_chdl */
EXPORTCH char *
MC_RetrieveAgentCode_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  char *retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_RetrieveAgentCode(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

/* MC_SearchForService_chdl */
EXPORTCH int
MC_SearchForService_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  int retval;
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  /* Args */
  const char* searchString;
  char*** agentNames;
  char*** serviceNames;
  int** agentIDs;
  int* numResults;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  searchString = Ch_VaArg(interp, ap, const char*);
  agentNames = Ch_VaArg(interp, ap, char***);
  serviceNames = Ch_VaArg(interp, ap, char***);
  agentIDs = Ch_VaArg(interp, ap, int**);
  numResults = Ch_VaArg(interp, ap, int*);

  retval = MC_SearchForService(
      temp_attr,
      searchString,
      agentNames,
      serviceNames,
      agentIDs,
      numResults
      );
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SemaphorePost_chdl */
EXPORTCH int
MC_SemaphorePost_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_SemaphorePost(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SemaphoreWait_chdl */
EXPORTCH int 
MC_SemaphoreWait_chdl(void *varg) /*{{{*/
{   
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;

  ChInterp_t interp;
  ChVaList_t ap;
  int id;
  int retval;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int );
  retval = MC_SemaphoreWait(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SendAgentMigrationMessage_chdl*/
EXPORTCH int
MC_SendAgentMigrationMessage_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  const char *message, *hostname;
  int port, retval;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  message = Ch_VaArg(interp, ap, char *);
  hostname = Ch_VaArg(interp, ap, char *);
  port = Ch_VaArg(interp, ap, int);
  retval = MC_SendAgentMigrationMessage(temp_attr, message, hostname, port);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SendAgentMigrationMessageFile_chdl*/
EXPORTCH int
MC_SendAgentMigrationMessageFile_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  char *filename, *hostname;
  int port, retval;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = NULL;

  Ch_VaStart(interp, ap, varg);
  filename = Ch_VaArg(interp, ap, char *);
  hostname = Ch_VaArg(interp, ap, char *);
  port = Ch_VaArg(interp, ap, int);
  retval = MC_SendAgentMigrationMessageFile(temp_attr, filename, hostname, port);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

EXPORTCH int
MC_SendSteerCommand_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  int retval;
  enum MC_SteerCommand_e command;
  extern mc_platform_p g_mc_platform;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  command = Ch_VaArg(interp, ap, enum MC_SteerCommand_e );
  retval = MC_SendSteerCommand(temp_attr, command);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SetAgentStatus_chdl */
EXPORTCH int
MC_SetAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgent_t agent;
  int status;
  int ret;
  ChInterp_t interp;
  ChVaList_t ap;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  status = Ch_VaArg(interp, ap, int);
  ret = MC_SetAgentStatus(agent, status);
  Ch_VaEnd(interp, ap);
  return ret;
} /*}}}*/

/* MC_SetDefaultAgentStatus_chdl */
EXPORTCH int
MC_SetDefaultAgentStatus_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int status;
  int ret;
  ChInterp_t interp;
  ChVaList_t ap;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(1););
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  status = Ch_VaArg(interp, ap, int);
  ret = MC_SetDefaultAgentStatus(temp_attr, status);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return ret;
} /*}}}*/

/* MC_SyncDelete_chdl*/
EXPORTCH int
MC_SyncDelete_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);
  retval = MC_SyncDelete(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_SyncInit_chdl*/
EXPORTCH int
MC_SyncInit_chdl(void *varg) /*{{{*/
{
  MCAgency_t temp_attr;
  extern mc_platform_p g_mc_platform;
  int retval;
  ChInterp_t interp;
  ChVaList_t ap;
  int id;

  temp_attr = (MCAgency_t)malloc(sizeof(struct agency_s));
  CHECK_NULL(temp_attr, exit(-1));
  temp_attr->mc_platform = g_mc_platform;

  Ch_VaStart(interp, ap, varg);
  id = Ch_VaArg(interp, ap, int);

  retval = MC_SyncInit(temp_attr, id);
  Ch_VaEnd(interp, ap);
  free(temp_attr);
  return retval;
} /*}}}*/

/* MC_TerminateAgent_chdl */
EXPORTCH int
MC_TerminateAgent_chdl(void *varg) /*{{{*/
{
  ChInterp_t interp;
  ChVaList_t ap;
  MCAgent_t agent;
  int retval;

  Ch_VaStart(interp, ap, varg);
  agent = Ch_VaArg(interp, ap, MCAgent_t);
  retval = MC_TerminateAgent(agent);
  Ch_VaEnd(interp, ap);
  return retval;
} /*}}}*/

